/* **********************************************************
 * Copyright 2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.tracing.otel;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.Message;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.internal.tracing.TracingScope;
import com.vmware.vapi.internal.tracing.TracingSpan;
import com.vmware.vapi.internal.util.StringUtils;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.std.StandardDataFactory;
import com.vmware.vapi.tracing.TracingLevel;

import io.opentelemetry.api.OpenTelemetry;
import io.opentelemetry.api.common.AttributeKey;
import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.StatusCode;
import io.opentelemetry.context.Context;
import io.opentelemetry.context.Scope;

/**
 * A class which wraps an OpenTelemetry {@link Span}. All method invocations are
 * ultimately forwarded to the wrapped {@code Span}.
 *
 * This wrapper takes care of checking whether (and how) method calls should be
 * forwarded to the wrapped OpenTelemetry {@link Span}.
 */
public class OtelTracingSpan implements TracingSpan {
    private static Logger logger = LoggerFactory.getLogger(OtelTracingSpan.class);

    private static final int MAX_ATTRIBUTE_LENGTH = 255;

    private final Span span;
    private final TracingLevel tracingLevel;
    private OpenTelemetry otel;

    /**
     * @param otel the OpenTelemetry instance that provides the tracing functionality
     * @param span the actual OpenTelemetry span to which the method invocation should be forwarded
     *        if the configuration allows it
     * @param tracingLevel indicates how detailed the tracing should be; for example, if this
     *        argument is {@link TracingLevel#INFO INFO}, method {@link #addEvent(String)} will be
     *        no-op
     */
    public OtelTracingSpan(OpenTelemetry otel, Span span,
                       TracingLevel tracingLevel) {
        Validate.notNull(span);
        Validate.notNull(tracingLevel);
        this.span = span;
        this.tracingLevel = tracingLevel;
        this.otel = otel;
    }

    /**
     * Forwards to {@link Span#makeCurrent()}
     *
     * @return this instance, for method chaining
     */
    @Override
    public TracingScope makeCurrent() {
        return new OtelTracingScope(span.makeCurrent());
    }

    /**
     * Forwards to {@link Span#setAttribute(AttributeKey, Object)}, but first
     * makes sure that the combined length of {@code key} and {@code value} does
     * not exceed {@link #MAX_ATTRIBUTE_LENGTH}. If it does exceed it, the value
     * may be truncated.
     */
    @Override
    public <T> TracingSpan setAttribute(TracingAttributeKey<T> key, T value) {
        if (key != null && value != null) {
            AttributeKey<T> attrKey = key.getAttributeKey();
            String sKey = attrKey.getKey();
            int keyLength = sKey.length();
            if (keyLength > MAX_ATTRIBUTE_LENGTH) {
                logger.warn("Tracing attribute key is longer than {}. Won't "
                        + "set the attribute into the span. key='{}', value='{}'",
                        MAX_ATTRIBUTE_LENGTH, sKey, value);
            } else {
                String sValue = value.toString();
                int maxLengthForValue = MAX_ATTRIBUTE_LENGTH - keyLength;
                if (sValue.length() > maxLengthForValue) {
                    String truncatedValue = sValue.substring(0, maxLengthForValue);
                    logger.warn("The combined length of attribute key and value "
                            + "exceeds the maximum length of {} mandated by Wavefront. "
                            + "Truncated the value to fit within the maximum length. "
                            + "key='{}', value='{}', truncatedValue='{}'",
                            MAX_ATTRIBUTE_LENGTH, sKey, value, truncatedValue);
                    span.setAttribute(sKey, truncatedValue);
                } else {
                    span.setAttribute(attrKey, value);
                }
            }
        }
        return this;
    }

    @Override
    public TracingSpan setStatusOk() {
        logger.trace("span status set to ok");
        span.setStatus(StatusCode.OK);
        return this;
    }

    @Override
    public TracingSpan setStatusError(String errorType, String errorMessage) {
        logger.trace("span status set to error");
        if (span.isRecording()) {
            span.setStatus(StatusCode.ERROR);

            // If we explicitly set the 'error' attribute to 'true', and if we then
            // export the spans to Jaeger, it complains that there's a duplicate tag
            // 'error:true'. If we don't set this attribute, the Jaeger UI still
            // shows 'error:true', and doesn't complain. It's not certain if this is
            // a feature or a bug of Jaeger, but, at least for the time being, we
            // won't set the 'error' attribute to 'true' explicitly:
            // _span.setAttribute(TracingAttributeKey.ERROR.getAttributeKey(), true);

            if (StringUtils.isNotBlank(errorType)) {
                setAttribute(TracingAttributeKey.ERROR_TYPE, errorType);
            }
            // We're not setting the below attribute directly into the span, but
            // we go through our own setAttribute() method, because the error
            // message may be too long and we might exceed the maximum of 255
            // characters. (This maximum is imposed by Wavefront.) If we go
            // through our setAttribute() method, it will trim the value, if
            // necessary.
            if (StringUtils.isNotBlank(errorMessage)) {
                setAttribute(TracingAttributeKey.ERROR_MESSAGE, errorMessage);
            }
        }
        return this;
    }

    @Override
    public TracingSpan setStatusError(Throwable exception) {
        String exceptionType = (exception != null
                ? exception.getClass().getCanonicalName() : null);
        String exceptionMessage = (exception != null
                ? exception.getMessage() : null);
        setStatusError(exceptionType, exceptionMessage);
        recordException(exception);
        return this;
    }

    @Override
    public TracingSpan setStatusError(ErrorValue errorValue) {
        if (span.isRecording()) {
            List<Message> msgs = StandardDataFactory.getMessagesFromErrorValue(errorValue);
            if (!msgs.isEmpty()) {
                Message msg = msgs.get(0);
                setStatusError(errorValue.getName(), msg.getDefaultMessage());
                if (tracingLevel == TracingLevel.TRACE) {
                    // add all subsequent error messages as events
                    for (int i = 1; i < msgs.size(); i++) {
                        Message m = msgs.get(i);
                        span
                        .addEvent(TracingAttributeKey.ERROR.getAttributeKey().getKey(),
                                  Attributes.of(TracingAttributeKey.ERROR_TYPE.getAttributeKey(),
                                                m.getId(),
                                                TracingAttributeKey.ERROR_MESSAGE.getAttributeKey(),
                                                m.getDefaultMessage()));
                    }
                }
            } else {
                setStatusError(errorValue.getName(), null);
            }
        }
        return this;
    }

    private TracingSpan recordException(Throwable ex) {
        if (ex != null && tracingLevel == TracingLevel.TRACE) {
            span.recordException(ex);
        }
        return this;
    }

    @Override
    public TracingSpan addEvent(String name) {
        if (isTracingLevelAtLeast(TracingLevel.DEBUG)) {
            span.addEvent(name, Attributes.empty());
        }
        return this;
    }

    @Override
    public <T> TracingSpan addEvent(String name, TracingAttributeKey<T> key, T value) {
        if (isTracingLevelAtLeast(TracingLevel.DEBUG)) {
            Attributes attrs = (key != null
                ? Attributes.of(key.getAttributeKey(), value) : Attributes.empty());
            span.addEvent(name, attrs);
        }
        return this;
    }

    @Override
    public void end() {
        logger.trace("span ended");
        if (span.isRecording()) {
            span.end();
        }
    }

    // DEV NOTE: currently, we propagate only the *span* data; however, in
    // the future we may decide to also propagate some Baggage and other
    // stuff. In such case, we will need to make this method take either a
    // Context argument, or an additional Baggage argument
    @Override
    public <C> TracingSpan injectInto(C carrier, KeyValueSetter<C> setter) {
            // Below, instead of making the span current (_span.makeCurrent())
            // and then, inside the scope, calling Context.current(), we could
            // normally do Context.current().with(_span).
            // But we can't do it here because, if OpenTelemetry is not on the
            // class path, Java throws
            // java.lang.ClassNotFoundException: io.opentelemetry.context.ImplicitContextKeyed
            // at runtime. However, it doesn't throw it when the code is like
            // below, so we'll stick with makeCurrent() and Context.current()
            try (Scope scope = span.makeCurrent()) {
                otel.getPropagators()
                        .getTextMapPropagator()
                        .inject(Context.current(),
                                carrier,
                                (c, key, value) -> setter.set(c, key, value));
            }
        return this;
    }

    private boolean isTracingLevelAtLeast(TracingLevel level) {
        return tracingLevel.ordinal() >= level.ordinal();
    }

    @Override
    public TracingSpan updateName(String newName) {
        newName = truncateLeft(newName);
        span.updateName(newName);
        return this;

    }

    private String truncateLeft(String name) {
        if (name != null && name.length() > MAX_ATTRIBUTE_LENGTH) {
            return name.substring(name.length() - MAX_ATTRIBUTE_LENGTH,
                                  name.length());
        }
        return name;
    }

    Span getDecoratedSpan() {
        return span;
    }
}
