/* **********************************************************
 * Copyright (c) 2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.util;

import org.apache.http.HttpInetConnection;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.conn.ManagedHttpClientConnection;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.internal.protocol.client.rpc.http.ApacheHttpUtil;
import com.vmware.vapi.internal.tracing.TracingSpan;
import com.vmware.vapi.internal.tracing.TracingSpan.KeyValueSetter;
import com.vmware.vapi.internal.tracing.otel.TracingAttributeKey;
import com.vmware.vapi.internal.tracing.otel.TracingAttributeKey.IoType;
import com.vmware.vapi.protocol.HttpConfiguration.Protocol;

import io.opentelemetry.context.propagation.TextMapPropagator;
import io.opentelemetry.context.propagation.TextMapSetter;

/**
 * Utility methods related to tracing
 */
public class TracingUtil {

    private static Logger _logger = LoggerFactory.getLogger(TracingUtil.class);

    private TracingUtil() {}

    /**
     * Sets information about the given {@code request} into the given
     * {@code span}. This method should be called right before sending
     * the request to the server, because this method also stores an
     * event into the span and the event indicates that the request is
     * just about to be sent.
     *
     * @param span
     *        the span in which to store information about the request
     *
     * @param request
     *        the request about which to store information into the span
     *
     * @param protocol
     *        the vAPI protocol which will be used for the request
     *
     * @param ioType
     *        indicates the I/O type, e.g. NIO
     */
    public static void registerRequestDataIntoSpan(TracingSpan span,
                                                   HttpUriRequest request,
                                                   Protocol protocol,
                                                   IoType ioType) {
        span.setAttribute(TracingAttributeKey.WIRE_PROTOCOL, protocolName(protocol))
            .setAttribute(TracingAttributeKey.HTTP_URL, request.getURI().toASCIIString())
            .setAttribute(TracingAttributeKey.HTTP_METHOD, request.getMethod())
            .setAttribute(TracingAttributeKey.IO, ioType.name())
            .addEvent("vAPI client >> peer",
                    TracingAttributeKey.THREAD_NAME, Thread.currentThread().getName());
    }

    /**
     * Sets information about the given {@code response} into the given
     * {@code span}. This method should be called right after receiving
     * the response from the server, because this method also stores an
     * event into the span and the event indicates that the response has
     * just been received.
     *
     * @param span
     *        the span in which to store information about the response
     *
     * @param response
     *        the response about which to store information into the span
     *
     * @param ctx
     *        the HTTP context for the Apache HTTP client
     */
    public static void registerResponseIntoSpan(TracingSpan span,
                                                 HttpResponse response,
                                                 HttpContext ctx) {
        span.addEvent("vAPI client << peer",
                TracingAttributeKey.THREAD_NAME, Thread.currentThread().getName())
            .setAttribute(TracingAttributeKey.HTTP_STATUS_CODE,
                (long) response.getStatusLine().getStatusCode());

        String contentType = ApacheHttpUtil.getContentType(response);
        if (contentType != null) {
            span.setAttribute(TracingAttributeKey.HTTP_CONTENT_TYPE, contentType);
        }

        // try to extract the local (ephemeral) port from the connection;
        // if we can set that attribute, it may enable us to correlate the
        // network traffic between the peers; for example, Envoy always
        // logs the ports, and if we also log the port, we can correlate
        // between our log and the Envoy log and thus check if some
        // connections are failing
        int localPort = -2;
        try {
            if (ctx instanceof HttpCoreContext) {
                HttpInetConnection connection = ((HttpCoreContext) ctx)
                       .getConnection(HttpInetConnection.class);
                if (connection != null) {
                    localPort = connection.getLocalPort();
                }
            } else if (ctx instanceof ManagedHttpClientConnection) {
                localPort = ((ManagedHttpClientConnection) ctx).getSocket().getLocalPort();
            }

        } catch (Exception e) {
            // ignore; below, we just won't add a span attribute about the local port
        }
        if (localPort != -2) {
            span.setAttribute(TracingAttributeKey.NET_HOST_PORT, (long) localPort);
        } else {
            _logger.trace("Unable to extract local port from HTTP client");
        }
    }

    /**
     * Sets into the given {@code request} a header whose name is the given
     * {@code name} and whose value is the given {@code value}.
     *
     * <p>Arguably, in and of itself, this method is not very useful, but we
     * need it to act as a {@link TextMapSetter} or a {@link KeyValueSetter}.
     * For example, we could do:</p>
     *
     * <blockquote><code><pre>
     * openTelemetry.getPropagators().getTextMapPropagator()
     *          .inject(currentContext, httpRequest, TracingUtil::addHeader)
     * </pre></code></blockquote>
     *
     * @param request
     *        the HTTP request to which a header must be added; if {@code
     *        null}, this method will do nothing
     *
     * @param name
     *        the name of the header which must be added to the given {@code
     *        request}; if {@code null}, this method will do nothing
     *
     * @param value
     *        the value of the header which must be added to the given {@code
     *        request}
     *
     * @see TextMapPropagator#inject(io.opentelemetry.context.Context, Object, TextMapSetter)
     * @see KeyValueSetter
     */
    public static void addHeader(HttpUriRequest request, String name, String value) {
        if (request != null && name != null) {
            request.addHeader(name, value);
        }
    }

    private static String protocolName(Protocol protocol) {
       switch(protocol.name()) {
           case "VAPI_JSON_RPC_1_0":
           case "VAPI_JSON_RPC_1_1":
               return "jsonrpc";
           default:
               //Shouldn't reach this point
               return "";
       }
    }
}
