/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.util.async;

import java.io.PrintWriter;
import java.io.StringWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.data.DataValue;

/**
 * Decorator which guards an existing handle against method calls after the
 * operation associated with the handle has been completed (i.e. multiple calls
 * to {@link #setError(RuntimeException)} or {@link #setResult(Object)}).
 *
 * @param <T> type of the operation output
 */
public final class StrictAsyncHandle<T> extends AsyncHandle<T> {

    private static final Logger logger = LoggerFactory.getLogger(StrictAsyncHandle.class);

    private final AsyncHandle<T> decorated;
    private Object lock = new Object();
    private boolean completed = false;
    private StackTraceElement[] completedAt;

    /**
     * Creates a decorator around the specified handle.
     *
     * @param decorated handle to be decorated
     */
    public StrictAsyncHandle(AsyncHandle<T> decorated) {
        this.decorated = decorated;
    }

    @Override
    public void updateProgress(DataValue progress) {
        synchronized(lock) {
            assertNotCompleted();
            decorated.updateProgress(progress);
        }
    }

    @Override
    public void setResult(T result) {
        synchronized (lock) {
            assertNotCompleted();
            decorated.setResult(result);
            markCompleted();
        }
    }

    @Override
    public void setError(RuntimeException error) {
        synchronized (lock) {
            assertNotCompleted();
            decorated.setError(error);
            markCompleted();
        }
    }

    /**
     * Marks the response as completed to prevent setting the result twice
     */
    private void markCompleted() {
        completed = true;
        completedAt = Thread.currentThread().getStackTrace();
    }

    /**
     * Verifies that the operation is not yet completed.
     *
     * @throws IllegalStateException if the operation is already completed
     */
    private void assertNotCompleted() {
        if (completed) {
            if (logger.isErrorEnabled()) {
                String currentStackTrace = getStackTrace(new Throwable());
                Throwable prevTrace = new Throwable();
                prevTrace.setStackTrace(completedAt);
                String prevStrackTrace = getStackTrace(prevTrace);
                String newLine = System.getProperty("line.separator");
                logger.error("Second attempt to complete the request" + newLine +
                        currentStackTrace + newLine +
                        "First attempt to complete the request" + newLine +
                        prevStrackTrace);
            }
            throw new IllegalStateException("Operation is already completed");
        }
    }

    private static String getStackTrace(Throwable throwable) {
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw, true);
        throwable.printStackTrace(pw);
        return sw.getBuffer().toString();
    }
}
