/* **********************************************************
 * Copyright 2014 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.util.io;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * {@link FilterInputStream} decorator that releases the stream reference after
 * the stream has been closed.
 *
 * @author Anton Atanassov
 *
 */
public class ReleasableInputStream extends FilterInputStream {

    /**
     * Directly calls the constructor of {@link FilterInputStream}
     *  with the same parameters.
     */
    public ReleasableInputStream(InputStream in) {
        super(in);
    }

    /**
     * This stream does not support mark and reset.
     *
     * @return - always <code>false<code>
     */
    @Override
    public boolean markSupported() {
        return false;
    }

    /**
     * This stream does not support mark and reset.
     *
     * The method does nothing.
     */
    @Override
    public void mark(int readAheadLimit) {
        // do nothing
    }

    /**
     * This stream does not support mark and reset.
     *
     * Always throws an IOException.
     *
     * @throws IOException
     */
    @Override
    public void reset() throws IOException {
        throw new IOException("mark/reset not supported");
    }

    /**
     * closes the stream and releases its reference.
     */
    @Override
    public synchronized void close() throws IOException {
        super.close();
        // release the reference to the initial stream
        in = new ExhaustedInputStream();
    }

    /**
     * An empty input stream that always behaves as if its end has been reached.
     *
     * @author Anton Atanassov
     *
     */
    private static final class ExhaustedInputStream extends InputStream {

        @Override
        public int read() {
            return -1;
        }

    }
}
