/* **********************************************************
 * Copyright (c) 2013, 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.l10n;

import java.util.Calendar;

import com.vmware.vapi.bindings.convert.ConverterException;
import com.vmware.vapi.internal.util.Rfc3339DateTimeConverter;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.std.LocalizableMessage;

/**
 * Converter for arguments of <code>LocalizableMessage</code>.
 *
 * @see LocalizableMessage#setArgs(java.util.List)
 */
public class MessageArgConverter {

    public Calendar toCalendarArgument(String arg) {
        Validate.notNull(arg);
        try {
            // this is what we use to convert Calendar to String
            // in bindings layer
            return new Rfc3339DateTimeConverter().fromStringValue(arg);
        } catch (ConverterException ex) {
            throw new LocalizationException(
                    "Can't convert Calendar message argument", ex);
        }
    }

    public String fromCalendar(Calendar arg) {
        Validate.notNull(arg);
        try {
            return new Rfc3339DateTimeConverter().toStringValue(arg);
        } catch (ConverterException ex) {
            throw new LocalizationException(
                    "Can't convert Calendar message argument", ex);
        }
    }

    public Long toLong(String arg) {
        try {
            // this is what we use for doubles in our JSON serialization
            // as well
            return Long.valueOf(arg);
        } catch(NumberFormatException ex) {
            throw new LocalizationException(
                    "Can't convert Long message argument", ex);
        }
    }

    public String fromLong(Long arg) {
        try {
            return Long.toString(arg);
        } catch(NumberFormatException ex) {
            throw new LocalizationException(
                    "Can't convert Long message argument", ex);
        }
    }

    public Double toDouble(String arg) {
        try {
            return Double.valueOf(arg);
        } catch(NumberFormatException ex) {
            throw new LocalizationException(
                    "Can't convert Double message argument", ex);
        }
    }

    public String fromDouble(Double arg) {
        try {
            // this is what we use for doubles in our JSON serialization
            // as well
            return Double.toString(arg);
        } catch(NumberFormatException ex) {
            throw new LocalizationException(
                    "Can't convert Double message argument", ex);
        }
    }

    public String[] convertToStrings(Object[] args) {
        if (args == null) {
            return new String[0];
        }

        String[] result = new String[args.length];
        for (int i = 0; i < args.length; i++) {
            result[i] = convertArgToString(args[i]);
        }

        return result;
    }

    String convertArgToString(Object arg) {
        Validate.notNull(arg);

        if (arg.getClass() == Long.class) {
            return fromLong((Long) arg);
        }

        if (arg.getClass() == Double.class) {
            return fromDouble((Double) arg);
        }

        if (arg.getClass() == String.class) {
            return (String) arg;
        }

        if (arg instanceof Calendar) {
            return fromCalendar((Calendar) arg);
        }

        throw new LocalizationException(
                "Unsupported type for message argument: " + arg.getClass());
    }
}
