/* **********************************************************
 * Copyright 2013-2016, 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.protocol;

import java.net.URI;
import java.net.URISyntaxException;
import java.security.KeyStore;
import java.util.Map;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.core.ApiProviderStub;
import com.vmware.vapi.core.ApiProviderStubImpl;
import com.vmware.vapi.internal.protocol.ExecutorApiProvider;
import com.vmware.vapi.internal.protocol.client.msg.json.JsonApiProvider;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient;
import com.vmware.vapi.internal.protocol.client.rpc.http.ApacheHttpAsyncClientTransport;
import com.vmware.vapi.internal.protocol.client.rpc.http.HttpClient;
import com.vmware.vapi.internal.protocol.common.json.JsonDeserializer;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.HttpConfiguration.LibType;
import com.vmware.vapi.protocol.HttpConfiguration.SslConfiguration;

/**
 * Factory for creating {@link ProtocolConnection} instances which uses JSON for
 * messaging protocol.
 */
public class JsonProtocolConnectionFactory implements ProtocolConnectionFactory {

    // Default HTTP/HTTPS ports. As described in RFC2616 (section 3.2.2) and
    // RFC2818 (section 2.3) these ports should be used if the user doesn't
    // specify port for the HTTP connection.
    private static int DEFAULT_HTTPS_PORT = 443;
    private static int DEFAULT_HTTP_PORT = 80;

    public enum Protocol {
        http
    }

    public JsonProtocolConnectionFactory() {
    }

    @Override
    public ProtocolConnection getConnection(String type, String args,
            KeyStore trustStore) {
        if (type.equals(Protocol.http.toString())) {
            SslConfiguration sslConfig =
                    new SslConfiguration.Builder(trustStore).getConfig();
            return getHttpConnection(args, null, new HttpConfiguration.
                    Builder().setSslConfiguration(sslConfig).getConfig());
        } else {
            return null;
        }
    }

    @Override
    public ProtocolConnection getHttpConnection(String uri,
                                                ClientConfiguration clientConfig,
                                                HttpConfiguration httpConfig) {
        CorrelatingClient transport = createCorrelatingClient(uri,
                                                              clientConfig,
                                                              httpConfig);
        Map<String, JsonDeserializer> deserializers;
        deserializers = JsonApiProvider.prepareDeserializers(clientConfig);
        ApiProvider provider = new JsonApiProvider(transport,
                                                   clientConfig,
                                                   deserializers);
        if (clientConfig != null && clientConfig.getExecutor() != null
            && httpConfig != null && !httpConfig.getLibraryType()
                    .equals(LibType.APACHE_HTTP_ASYNC_CLIENT)) {
            provider = new ExecutorApiProvider(provider,
                                               clientConfig.getExecutor());
        }
        return new ProtocolConnectionImpl(transport, provider);
    }

    @Override
    public ProtocolConnection getInsecureConnection(String type, String args) {
        if (type.equals(Protocol.http.toString())) {
            return getHttpConnection(args, null, null);
        } else {
            return null;
        }
    }

    /**
     * Set default details to the endpoint URI. Such detail is the port. If the
     * URI doesn't specify port a default port is used based on the specified
     * schema. This is as described in in RFC2616 section 3.2.2 and RFC2818
     * section 2.3.
     *
     * @param uri URI to be checked for missing details; must not be null
     * @return uri with added default details
     * @throws IllegalArgumentException if the argument is null or invalid URI
     */
    private static String setDefaultHTTPConnectionDetails(String uri) {
        Validate.notNull(uri, "URI is required for creating client");
        try {
            URI connUri = new URI(uri);

            // Check if there is no port specified or the URI invalid ('_' is
            // not valid character and leads to unparsable URI (getPort returns
            // always -1).
            if (-1 != connUri.getPort() ||
                connUri.getAuthority() == null ||
                connUri.getAuthority().contains("_") ||
                connUri.getAuthority().contains(":")) {
                return uri;
            }

            int port = -1;
            // Set the default protocol port in the uri
            if ("https".equalsIgnoreCase(connUri.getScheme())) {
                port = DEFAULT_HTTPS_PORT;
            } else if ("http".equalsIgnoreCase(connUri.getScheme())) {
                port = DEFAULT_HTTP_PORT;
            } else {
                throw new IllegalArgumentException(
                        "No port specified and no default port " +
                        "found for schema " + connUri.getScheme());
            }

            return new URI(connUri.getScheme(),
                           connUri.getUserInfo(),
                           connUri.getHost(),
                           port,
                           connUri.getPath(),
                           connUri.getQuery(),
                           connUri.getFragment()).toString();
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException(
                    "Unable to parse the specified URI " + uri,
                    e);
        }
    }

    private CorrelatingClient createCorrelatingClient(String uri,
                                                      ClientConfiguration clientConfig,
                                                      HttpConfiguration httpConfig) {
        if (httpConfig == null) {
            httpConfig = new HttpConfiguration.Builder().getConfig();
        }

        uri = setDefaultHTTPConnectionDetails(uri);
        return createHttpTransport(uri,
                                   clientConfig,
                                   httpConfig);
    }

    /**
     * Factory method which creates the <code>CorrelatingClient</code> to use
     * as transport for the <code>ProtocolConnection</code>s created by this
     * factory.
     */
    protected CorrelatingClient createHttpTransport(String uri,
                                                    ClientConfiguration clientConfig,
                                                    HttpConfiguration httpConfig) {
        switch (httpConfig.getLibraryType()) {
        case APACHE_HTTP_CLIENT:
            return new HttpClient(uri, httpConfig);
        case APACHE_HTTP_ASYNC_CLIENT:
            return new ApacheHttpAsyncClientTransport(
                        uri,
                        httpConfig);
        default:
            throw new IllegalArgumentException("Unsupported library type "
                    + httpConfig.getLibraryType());
        }
    }

    /**
     * This class is a container for vAPI transport and messaging implementations
     * that combines them into a single ApiProvider.
     */
    private static final class ProtocolConnectionImpl implements ProtocolConnection {

        private final CorrelatingClient transport;
        private final ApiProvider provider;

        /**
         * TODO this constructor should ideally accept messaging abstractions
         *      instead of ApiProvider, but this will be implemented in a
         *      subsequent refactoring CL.
         *
         * @param transport a vAPI transport implementation. cannot be null.
         * @param provider a vAPI ApiProvider implementation. cannot be null.
         */
        private ProtocolConnectionImpl(CorrelatingClient transport, ApiProvider provider) {
            Validate.notNull(transport);
            Validate.notNull(provider);
            this.transport = transport;
            this.provider = provider;
        }

        @Override
        public void disconnect() {
            transport.close();
        }

        @Override
        public ApiProvider getApiProvider() {
            return provider;
        }

        @Override
        public ApiProviderStub getApiProviderStub() {
            return new ApiProviderStubImpl(getApiProvider());
        }
    }
}
