/* **********************************************************
 * Copyright 2013-2014 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.protocol;

import com.vmware.vapi.core.ApiProvider;

public interface ProtocolHandlerFactory {

    /**
     * Create protocol handler.<p>
     * Protocol handler has a Http Endpoint if args is not null<p>
     * Protocol handler has a Https Endpoint if secureArgs is not null. A valid
     * keystore configuration must be provided in this case<p>
     *
     * @param type protocol type; must not be <code>null</code>
     * @param args http uri; might be <code>null</code>
     * @param secureArgs https uri; might be <code>null</code>
     * @param keystore the keystore, which contains the server certificates and
     *            keys; must be not <code>null</code> if secureArgs is provided
     * @param keystorePassword the keystore password; might be <code>null</code>
     * @param keyPassword the key password, if different from the keystore
     *            password; might be <code>null</code>
     * @param provider the API provider; must not be <code>null</code>
     * @return the protocol handler
     */
    ProtocolHandler getProtocolHandler(String type,
                                       String args,
                                       String secureArgs,
                                       String keystore,
                                       String keystorePassword,
                                       String keyPassword,
                                       ApiProvider provider);

    /**
     * Create protocol handler with configurable client authentication.<p>
     * Protocol handler has a Http Endpoint if args is not null<p>
     * Protocol handler has a Https Endpoint if secureArgs is not null. A valid
     * keystore configuration must be provided in this case. A valid truststore
     * configuration must be provided client authentication<p>
     *
     * @param type protocol type; must not be <code>null</code>
     * @param args http uri; might be <code>null</code>
     * @param secureArgs https uri; might be <code>null</code>
     * @param clientAuth the client authentication to be used: true to require a
     *            valid certificate from the client to accept a connection and
     *            fail if the certificate is not supplied, want to request the
     *            client to supply a valid certificate, but do not fail if is
     *            not provided, false to not require a client certificate; might
     *            be <code>null</code>
     * @param keystoreConfig the keystore configuration; must be not
     *            <code>null</code> if secureArgs is provided
     * @param truststoreConfig the trustore configuration; must be not
     *            <code>null</code> if client authentication is true
     * @param provider the API provider; must not be <code>null</code>
     * @return the protocol handler
     */
    ProtocolHandler getProtocolHandler(String type,
                                       String args,
                                       String secureArgs,
                                       String clientAuth,
                                       KeyStoreConfig keystoreConfig,
                                       TrustStoreConfig truststoreConfig,
                                       ApiProvider provider);

    /**
     * Create a insecure protocol handler.
     *
     * @param type protocol type; must not be <code>null</code>
     * @param args http uri; might be <code>null</code>
     * @param provider the API provider; must not be <code>null</code>
     * @return the protocol handler
     */
    ProtocolHandler getInsecureProtocolHandler(String type,
                                               String args,
                                               ApiProvider provider);

    /**
     * This class contains the keystore configuration. The class is immutable.
     */
    public final class KeyStoreConfig {
        private final String path;
        private final String password;
        private final String keyPassword;

        /**
         * Creates a keystore configuration.
         *
         * @param path the keystore path
         * @param password the keystore password
         * @param keyPassword the key password, if is different than keystore
         *            password, or null if is the same password
         */
        public KeyStoreConfig(String path, String password, String keyPassword) {
            this.path = path;
            this.password = password;
            this.keyPassword = keyPassword;
        }

        public String getPath() {
            return path;
        }

        public String getPassword() {
            return password;
        }

        public String getKeyPassword() {
            return keyPassword;
        }
    }

    /**
     * This class contains the truststore configuration. The class is immutable.
     */
    public final class TrustStoreConfig {
        private final String path;
        private final String password;

        /**
         * Creates a truststore configuration..
         *
         * @param path the truststore path
         * @param password the truststore password
         */
        public TrustStoreConfig(String path, String password) {
            this.path = path;
            this.password = password;
        }

        public String getPath() {
            return path;
        }

        public String getPassword() {
            return password;
        }
    }
}
