/* **********************************************************
 * Copyright (c) 2012, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.protocol;

import java.util.Map;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.ExecutionContext.SecurityContext;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.internal.util.Validate;

/**
 * Implementations of this interface might be attached to client
 * {@link ApiProvider} implementations as request post-processors or to the
 * server {@link ApiProvider} implementations as request pre-processors.
 */
public interface RequestProcessor {

    public static final String UTF8_CHARSET = "UTF-8";
    public static final String SECURITY_CONTEXT_KEY = "security_context";

    /**
     * This key is used to transfer the authentication results to the
     * authentication handlers using the metadata parameter. The object
     * corresponding to the key is expected to be a {@link Map} with
     * {@link String} key and {@link Object} value. Content of the map is
     * authentication handler specific.
     */
    public static final String SECURITY_PROC_METADATA_KEY =
            "security_processor_metadata";

    /**
     * Processes and possibly modifies the provided request byte array
     *
     * @param requestBytes must not be null. If text is passed as it MUST be UTF-8
     *                     encoded.
     * @param metadata provides additional metadata to the processor that will be
     *                 passed to the rest of the processor chain. Must not be null.
     *                 The content under the
     *                 {@link RequestProcessor#SECURITY_PROC_METADATA_KEY} key
     *                 will be put in the request's {@link SecurityContext} under
     *                 the same key i.e. will be accessible by the
     *                 authentication handlers.
     * @param request the vapi request structure. Must not be null.
     * @return the result of the processor. Must not be null. If text is returned
     *         it MUST be UTF-8 encoded.
     */
    byte[] process(byte[] requestBytes,
                   Map<String, Object> metadata,
                   Request request);

    /**
     * This class represents a vAPI request.
     */
    public class Request {
        private final String serviceId;
        private final String operationId;
        private final ExecutionContext ctx;
        private final DataValue input;

        /**
         * @param serviceId must not be <code>null</code>
         * @param operationId must not be <code>null</code>
         * @param ctx can be <code>null</code>
         * @param input must not be <code>null</code>
         */
        public Request(String serviceId,
                       String operationId,
                       ExecutionContext ctx,
                       DataValue input) {
            Validate.notNull(serviceId);
            Validate.notNull(operationId);
            Validate.notNull(input);
            this.serviceId = serviceId;
            this.operationId = operationId;
            this.ctx = ctx;
            this.input = input;
        }

        /**
         * @return the serviceId. must not be <code>null</code>
         */
        public String getServiceId() {
            return serviceId;
        }

        /**
         * @return the operationId. must not be <code>null</code>
         */
        public String getOperationId() {
            return operationId;
        }

        /**
         * @return the execution context. can be <code>null</code>
         */
        public ExecutionContext getCtx() {
            return ctx;
        }

        /**
         * @return the data value. must not be <code>null</code>
         */
        public DataValue getInput() {
            return input;
        }
    }

}
