/* **********************************************************
 * Copyright 2009 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.protocol.server.rpc.http.impl;

import com.vmware.vapi.protocol.server.rpc.http.Endpoint;


/**
 * Implementation of an HTTP server endpoint.
 */

public class HttpEndpoint implements Endpoint {

   protected String _host;

   protected int _port;

   protected Protocol _protocol;

   protected int _numAcceptors;

   protected int _acceptQueueSize;

   protected int _maxIdleTime;

   protected EndpointType _type = EndpointType.BLOCKING;

   /**
    * Construct an HTTP endpoint for a particular host, port, and protocol.
    *
    * @param host The listening interface.
    * @param port The listening port.
    * @param protocol The listening protocol.
    */
   public HttpEndpoint(String host, int port, Protocol protocol) {
      _host = host;
      _port = port;
      _protocol = protocol;
      _numAcceptors = 1;
      _acceptQueueSize = 0;
      _maxIdleTime = 200000;
   }

   /**
    * Construct an HTTP endpoint that listens on all interfaces.
    *
    * @param port The listening port.
    */
   public HttpEndpoint(int port) {
      this("0.0.0.0", port, Protocol.HTTP);
   }


   /**
    * Construct an HTTP endpoint for a particular host and port.
    *
    * @param host The listening interface.
    * @param port The listening port.
    */
   public HttpEndpoint(String host, int port) {
      this (host, port, Protocol.HTTP);
   }

   public int getPort() {
      return _port;
   }

   public String getHost() {
      return _host;
   }

   public Protocol getProtocol() {
      return _protocol;
   }

   public String toString() {
      return _protocol.toString() + ":" + _host + ":" + Integer.toString(_port);
   }

   @Override
   public int getAcceptQueueSize() {
      return _acceptQueueSize;
   }

   @Override
   public int getNumAcceptors() {
      return _numAcceptors;
   }

   @Override
   public void setAcceptQueueSize(int acceptQueueSize) {
      _acceptQueueSize = acceptQueueSize;
   }

   @Override
   public void setNumAcceptors(int acceptors) {
      _numAcceptors = acceptors;
   }

   @Override
   public int getMaxIdleTime() {
       return _maxIdleTime;
   }

   @Override
   public void setMaxIdleTime(int maxIdleTime) {
       _maxIdleTime = maxIdleTime;
   }

   @Override
   public EndpointType getEndpointType() {
      return _type;
   }

   @Override
   public void setEndpointType(EndpointType type) {
      _type = type;
   }
}
