/* **********************************************************
 * Copyright 2009-2015, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.protocol.server.rpc.http.impl;

import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.server.rpc.http.SslEndpoint;

import org.springframework.core.io.Resource;

import java.io.IOException;

/**
 * Implementation of an HTTPS server endpoint.
 */
public class HttpsEndpoint extends HttpEndpoint implements SslEndpoint {

   private String _keyStorePath;

   private String _keyStorePassword;

   private String _keyPassword;

   private String _keyStoreType;

   private String _trustStorePath;

   private String _trustStorePassword;

   private boolean _needClientAuth;

   private boolean _wantClientAuth;

   private String _enabledProtocols;

   /**
    * Construct an SSL endpoint for a particular host, port, and protocol.
    *
    * @param host The listening interface.
    * @param port The listening port.
    */
   public HttpsEndpoint(String host, int port) {
      super(host, port, Protocol.HTTPS);
      _host = host;
      _port = port;
      _needClientAuth = false;
      _wantClientAuth = false;
      _enabledProtocols = "TLSv1,TLSv1.1,TLSv1.2";
   }

   /**
    * Construct an SSL endpoint that listens on all interfaces.
    *
    * @param port The listening port.
    */
   public HttpsEndpoint(int port) {
      this("0.0.0.0", port);
   }

   /**
    * Sets keystore type
    *
    * @param type keystore type
    */
   public void setKeyStoreType(String type) {
      _keyStoreType = type;
   }

   /**
    * Set the endpoint's keystore.
    * @param keyStorePath The path to the endpoint's keystore.
    */
   public void setKeyStorePath(String keyStorePath) {
      _keyStorePath = keyStorePath;
   }

   /**
    * Set keystore (for spring config).
    *
    * @param keystore resource representing the keystore. Must not be
    *           <code>null</code>. The resource must not be pointing to a file
    *           on the classpath which is in a jar.
    */
   public void setKeyStoreResource(Resource keystore) throws IOException {
      Validate.notNull(keystore);
      _keyStorePath = keystore.getFile().getAbsolutePath();
   }

   /**
    * Set the password for the keystore.
    * @param keyStorePassword The keystore password.
    */
   public void setKeyStorePassword(String keyStorePassword) {
      _keyStorePassword = keyStorePassword;
   }

   /**
    * Set the password for the private key.
    * @param keyPassword The key password.
    */
   public void setKeyPassword(String keyPassword) {
      _keyPassword = keyPassword;
   }

   /**
    * Set the endpoint's trust store.
    * @param trustStorePath The path to the endpoint's keystore.
    */
   public void setTrustStorePath(String trustStorePath) {
      _trustStorePath = trustStorePath;
   }

   /**
    * Set the password for the trust store.
    * @param trustStorePassword The keystore password.
    */
   public void setTrustStorePassword(String trustStorePassword) {
      _trustStorePassword = trustStorePassword;
   }

   @Override
   public String getKeyStorePath() {
      return _keyStorePath;
   }

   @Override
   public String getKeyStorePassword() {
      return _keyStorePassword;
   }

   @Override
   public String getKeyPassword() {
      return _keyPassword;
   }

   @Override
   public String getKeyAlias() {
      return "jetty"; // Jetty SSL connectors require "jetty" here
   }

   @Override
   public String getTrustStorePath() {
      return _trustStorePath;
   }

   @Override
   public String getTrustStorePassword() {
      return _trustStorePassword;
   }

   @Override
   public void setNeedClientAuth(boolean needed) {
      _needClientAuth = needed;
   }

   @Override
   public boolean getNeedClientAuth() {
      return _needClientAuth;
   }

   @Override
   public boolean getWantClientAuth() {
      return _wantClientAuth;
   }

   @Override
   public void setWantClientAuth(boolean wanted) {
      _wantClientAuth = wanted;
   }

   @Override
   public String getKeyStoreType() {
      return _keyStoreType;
   }

   /**
    * Sets the list of protocols that are enabled for use by SSL connections.
    *
    * <p>The specified protocols must be among {@code SSLv2Hello},
    * {@code SSLv3}, {@code TLSv1}, {@code TLSv1.1} and {@code TLSv1.2};
    * otherwise, applying this configuration over an {@link SSLSocket} later
    * would fail.
    *
    * <p>Defaults to {@code TLSv1,TLSv1.1,TLSv1.2}.
    *
    * @param enabledProtocols comma separated list of protocols enabled to use
    *        by SSL connections; must not be {@code null} or empty.
    */
   public void setEnabledProtocols(String enabledProtocols) {
     Validate.notEmpty(enabledProtocols);
     _enabledProtocols = enabledProtocols;
   }

   @Override
   public String getEnabledProtocols() {
     return _enabledProtocols;
   }
}
