/* **********************************************************
 * Copyright 2011-2016 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.protocol.server.rpc.http.impl;

import java.io.File;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.HttpConstraintElement;
import javax.servlet.ServletException;
import javax.servlet.ServletSecurityElement;
import javax.servlet.annotation.ServletSecurity.TransportGuarantee;

import org.apache.catalina.Context;
import org.apache.catalina.Executor;
import org.apache.catalina.LifecycleException;
import org.apache.catalina.LifecycleState;
import org.apache.catalina.Wrapper;
import org.apache.catalina.connector.Connector;
import org.apache.catalina.core.AprLifecycleListener;
import org.apache.catalina.core.StandardServer;
import org.apache.catalina.deploy.FilterDef;
import org.apache.catalina.deploy.FilterMap;
import org.apache.catalina.deploy.SecurityConstraint;
import org.apache.catalina.servlets.DefaultServlet;
import org.apache.catalina.startup.Tomcat;
import org.apache.coyote.AbstractProtocol;
import org.apache.coyote.ProtocolHandler;
import org.apache.coyote.http11.Http11NioProtocol;
import org.apache.coyote.http11.Http11Protocol;
import org.apache.tomcat.util.net.jsse.JSSEImplementation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.internal.protocol.server.rpc.http.util.FileUtil;
import com.vmware.vapi.internal.protocol.server.rpc.http.util.StringUtil;
import com.vmware.vapi.protocol.server.rpc.http.Endpoint;
import com.vmware.vapi.protocol.server.rpc.http.Filter;
import com.vmware.vapi.protocol.server.rpc.http.InternalException;
import com.vmware.vapi.protocol.server.rpc.http.Service;
import com.vmware.vapi.protocol.server.rpc.http.StaticContentService;

/**
 * Receives requests and passes them for processing through the
 * configured server processing steps. Once ready transmits
 * responses back to the client. Uses tcServer with Tomcat 7.0.
 */
public class TcServer extends AbstractServer {

   private static final Logger _logger =
           LoggerFactory.getLogger(TcServer.class);

   /** Connector property names */
   public static final String CONN_HOST = "address";
   public static final String CONN_NUM_ACCEPTORS = "acceptorThreadCount";
   public static final String CONN_ACCEPT_QUEUE_SIZE = "acceptCount";
   public static final String CONN_MAX_IDLE_TIME = "connectionTimeout";
   public static final String CONN_SSL_ENABLED = "SSLEnabled";
   public static final String CONN_KEYSTORE_TYPE = "keystoreType";
   public static final String CONN_KEYSTORE_FILE = "keystoreFile";
   public static final String CONN_KEYSTORE_PASS = "keystorePass";
   public static final String CONN_KEYSTORE_PROVIDER = "keystoreProvider";
   public static final String CONN_KEY_PASS = "keyPass";
   public static final String CONN_TRUSTSTORE_TYPE = "truststoreType";
   public static final String CONN_TRUSTSTORE_FILE = "truststoreFile";
   public static final String CONN_TRUSTSTORE_PASS = "truststorePass";
   public static final String CONN_TRUSTSTORE_PROVIDER = "truststoreProvider";
   public static final String CONN_CLIENT_AUTH = "clientAuth";
   public static final String CONN_ALGORITHM = "algorithm";
   public static final String CONN_CIPHERS = "ciphers";
   public static final String CONN_SSL_IMPLEMENTATION = "sslImplementationName";
   public static final String CONN_ENABLED_PROTOCOLS = "sslEnabledProtocols";
   public static final String SERVER = "server";
   public static final String ASYNC_TIMEOUT_KEY = "asyncTimeout";

   /** Magic value meaning timeout value not set. If timeout is not set, default value will never timeout */
   private static final long ASYNC_TIMEOUT_NOT_SET = -1000;

   /** HTTP server */
   private Tomcat _server;

   private Context _context;

   private String _staticContentBasePath;

   /** Thread pool to use with tcServer */
   private Executor _threadPool;

   /** Asynchronous timeout */
   private long _asyncTimeout = ASYNC_TIMEOUT_NOT_SET;

   public TcServer() {
      super();
   }

   /**
    * Set thread pool to be used by this server for handling
    * HTTP connections.
    *
    * @param threadPool the thread pool to use
    */
   public void setThreadPool(Executor threadPool) {
      _threadPool = threadPool;
   }

   /**
    * Set asynchronous timeout in millis
    *
    * @param asyncTimeout asynchronous timeout
    */
   public void setAsyncTimeout(long asyncTimeout) {
      _asyncTimeout = asyncTimeout;
   }

   void prepareToStart() throws Exception {
       _server = new Tomcat();

       if (_threadPool != null) {
          _server.getService().addExecutor(_threadPool);
       }

       createHandler();
       createConnectors();

       // Add AprLifecycleListener
       StandardServer server = (StandardServer) _server.getServer();
       AprLifecycleListener listener = new AprLifecycleListener();
       listener.setSSLEngine("on");
       server.addLifecycleListener(listener);

       applyServerConfigurator();

       if (_logger.isInfoEnabled()) {
          _logger.info("Starting server on " + Arrays.toString(_endpoints));
       }
   }

   @Override
   public void start() throws Exception {
       prepareToStart();
      _server.start();
   }

   private void createConnectors() {

      for (int i = 0; i < _endpoints.length; ++i) {
         Connector connector = null;

         if (_endpoints[i].getProtocol() == Endpoint.Protocol.HTTP) {
            connector = createHttpConnector((HttpEndpoint) _endpoints[i]);
            connector.setScheme("http");
         } else { // Endpoint.Protocol.HTTPS
            connector = createHttpsConnector((HttpsEndpoint) _endpoints[i]);
            connector.setScheme("https");
         }

         // common connector configuration
         connector.setProperty(CONN_HOST, _endpoints[i].getHost());
         connector.setPort(_endpoints[i].getPort());
         connector.setAttribute(CONN_NUM_ACCEPTORS, _endpoints[i].getNumAcceptors());
         connector.setAttribute(CONN_ACCEPT_QUEUE_SIZE, _endpoints[i].getAcceptQueueSize());
         connector.setAttribute(CONN_MAX_IDLE_TIME, _endpoints[i].getMaxIdleTime());
         connector.setAttribute(SERVER, "Apache");
         if (_threadPool != null) {
            ProtocolHandler pHandler = connector.getProtocolHandler();
            // cast to AbstractProtocol as setExecutor is not exposed in the interface
            if (pHandler instanceof AbstractProtocol) {
               ((AbstractProtocol) pHandler).setExecutor(_threadPool);
            } else {
              _logger.warn("Cannot inject custom Executor to the connector. " +
                           "A default Executor will be used");
            }
         }
         _server.getService().addConnector(connector);
         // First endpoint is also registered as server default connector
         if (i == 0) {
            _server.setConnector(connector);
         }
      }
   }

   private Connector createHttpConnector(HttpEndpoint endpoint) {
      switch (endpoint.getEndpointType()) {
         case NON_BLOCKING_NIO:
            return new Connector(Http11NioProtocol.class.getName());
         case BLOCKING:
            return new Connector();
         default:
            throw new IllegalArgumentException("Unknown Endpoint type: " +
                  endpoint.getEndpointType());
      }
   }

   /**
    * Creates connectors with a custom protocol implementation that exposes
    * tomcat endpoint trust manager
    * @param endpoint https server endpoint
    * @return a configured tomcat connector
    */
   private Connector createHttpsConnector(HttpsEndpoint endpoint) {
      Connector connector = null;
      switch (endpoint.getEndpointType()) {
         case NON_BLOCKING_NIO:
            connector = new Connector(Http11NioProtocol.class.getName());
            configureSslConnector(connector, endpoint);
            return connector;
         case BLOCKING:
            connector = new Connector(Http11Protocol.class.getName());
            configureSslConnector(connector, endpoint);
            return connector;
         default:
            throw new IllegalArgumentException("Unknown Endpoint type: " +
                  endpoint.getEndpointType());
      }
   }

   private void configureSslConnector(Connector connector,
                                      HttpsEndpoint endpoint) {

      connector.setSecure(true);
      connector.setProperty(CONN_SSL_ENABLED, "true");
      // The class name of the SSL implementation to use
      connector.setProperty(CONN_SSL_IMPLEMENTATION,
                            JSSEImplementation.class.getName());
      if (endpoint.getKeyStoreType() != null) {
         connector.setProperty(CONN_KEYSTORE_TYPE, endpoint.getKeyStoreType());
      }
      if (endpoint.getKeyStorePath() != null) {
         connector.setAttribute(CONN_KEYSTORE_FILE,
               FileUtil.getAbsoluteFilename(endpoint.getKeyStorePath()));
      }
      if (endpoint.getKeyStorePassword() != null) {
         connector.setAttribute(CONN_KEYSTORE_PASS, endpoint.getKeyStorePassword());
      }
      if (endpoint.getKeyPassword() != null) {
         connector.setAttribute(CONN_KEY_PASS, endpoint.getKeyPassword());
      }
      if (endpoint.getTrustStorePath() != null) {
         connector.setAttribute(CONN_TRUSTSTORE_FILE,
               FileUtil.getAbsoluteFilename(endpoint.getTrustStorePath()));
      }
      if (endpoint.getTrustStorePassword() != null) {
         connector.setAttribute(CONN_TRUSTSTORE_PASS, endpoint.getTrustStorePassword());
      }
      if (endpoint.getTrustStorePath() == null &&
          endpoint.getTrustStorePassword() == null) {
         connector.setProperty(CONN_TRUSTSTORE_FILE,
               (String) connector.getProperty(CONN_KEYSTORE_FILE));
         connector.setProperty(CONN_TRUSTSTORE_PASS,
               (String) connector.getProperty(CONN_KEYSTORE_PASS));
         connector.setProperty(CONN_TRUSTSTORE_TYPE,
               (String) connector.getProperty(CONN_KEYSTORE_TYPE));
      }

      // valid clientAuth values are: true, false, want
      if (endpoint.getNeedClientAuth()) {
         connector.setAttribute(CONN_CLIENT_AUTH, "true");
      } else {
         if (endpoint.getWantClientAuth()) {
            connector.setAttribute(CONN_CLIENT_AUTH, "want");
         } else {
            connector.setAttribute(CONN_CLIENT_AUTH, "false");
         }
      }
      // Configure ciphers allowed. Make sure that weak ciphers are excluded
      String[] cipherSuites = getEnabledSSLCiphers();
      connector.setProperty(CONN_CIPHERS, StringUtil.join(cipherSuites, ","));

      connector.setProperty(CONN_ENABLED_PROTOCOLS,
                            endpoint.getEnabledProtocols());
   }

   private void createHandler() throws ServletException {
      // Register default context path with static content directory
      // If static content servlet is not configured, registers default context path
      // with current directory.
      String baseDir = null;
      if (_staticContentBasePath != null) {
         baseDir = _staticContentBasePath;
      } else {
         baseDir = new File(".").getAbsolutePath();
      }

      _context = _server.addContext("/", baseDir);

      // register mime types
      registerMimeMappings();

      _context.setParentClassLoader(getClass().getClassLoader());

      // Add the services
      if (_services.length == 0) {
         throw new IllegalStateException("There are no services configured");
      }
      addServices();

      // Add the filters
      addFilters();
   }

   /**
    * Register mime mappings, creating a dummy context where Tomcat mime
    * mappings are registered by default, and copying mime mappings from dummy
    * context to Tomcat context. Then, the dummy context is stopped and
    * destroyed. mime types are registered using this trick because the list of
    * mime types is private.
    */
   private void registerMimeMappings() {
      try {
         Context dummyContext = _server.addContext("/dummyContext", ".");
         Tomcat.initWebappDefaults(dummyContext);
         String[] mimeMappings = dummyContext.findMimeMappings();
         for (String mimeMapping : mimeMappings) {
            _context.addMimeMapping(mimeMapping, dummyContext.findMimeMapping(mimeMapping));
         }
         dummyContext.stop();
         dummyContext.destroy();
      } catch (LifecycleException e) {
          _logger.error(e.getMessage(), e);
      }
   }

   /**
    * Adds services to server instance.
    */
   private void addServices() {
      int servletNo = 0;

      for (Service srv : _services) {

         String servletName = "servlet " +(servletNo++);
         Wrapper holder = Tomcat.addServlet(_context, servletName, srv.getServlet());

         // Add asynchronous timeout as servlet init parameter if value is set (can be overriden by
         // service configuration)
         if (_asyncTimeout != ASYNC_TIMEOUT_NOT_SET) {
            holder.addInitParameter(ASYNC_TIMEOUT_KEY, String.valueOf(_asyncTimeout));
         }

         if (srv.getInitParameters() != null) {
            for (Map.Entry<String, String> initParam : srv.getInitParameters().entrySet()) {
               holder.addInitParameter(initParam.getKey(), initParam.getValue());
            }
         }

         String path = srv.getPath();

         // For backward compatibility with the old multi-context version
         // additionally register all exact patterns with a trailing "/".
         _context.addServletMapping(path, servletName);
         if (path.indexOf('*') < 0 && !path.endsWith("/")) {
            String trailingSlashPath = path + "/";
            _context.addServletMapping(trailingSlashPath, servletName);
         }
         // support async for current servlet
         holder.setAsyncSupported(true);

         // add security constraints
         HttpConstraintElement dc = new HttpConstraintElement(
               resolveDataConstraint(srv.getTransportGuarantee()));
         SecurityConstraint[] securityConstraints = SecurityConstraint.createConstraints(
               new ServletSecurityElement(dc), path);

         for (SecurityConstraint securityConstraint : securityConstraints) {
            _context.addConstraint(securityConstraint);
         }

         // set servlet to initialize on startup (this makes init called on startup)
         holder.setLoadOnStartup(0);
      }
   }

   /**
    * Adds filters to server instance.
    */
   private void addFilters() {
      int filterNo = 0;
      for (Filter flt : _filters) {
         FilterDef filterDef = new FilterDef();
         String filterName = "filter " + (filterNo++);
         filterDef.setFilterName(filterName);
         filterDef.setFilter(flt.getFilter());
         if (flt.getInitParameters() != null) {
            for (Map.Entry<String, String> entry: flt.getInitParameters().entrySet()) {
               filterDef.addInitParameter(entry.getKey(), entry.getValue());
            }
         }
         _context.addFilterDef(filterDef);

         FilterMap filterMap = new FilterMap();
         filterMap.setFilterName(filterName);
         filterMap.addURLPattern(flt.getPath());
         for (Filter.Dispatcher t : flt.getDispatchers()) {
            filterMap.setDispatcher(t.name());
         }
         _context.addFilterMap(filterMap);
      }
   }

   private static TransportGuarantee resolveDataConstraint(Service.TransportGuarantee tg) {
      switch (tg) {
         case NONE:
            return TransportGuarantee.NONE;

         case INTEGRAL:
            return TransportGuarantee.CONFIDENTIAL;

         case CONFIDENTIAL:
            return TransportGuarantee.CONFIDENTIAL;

         default:
            throw new InternalException("Unknown transport guarantee " + tg);
      }
   }

   @Override
   public void stop() throws Exception {

      shutdown();

      if (_logger.isInfoEnabled()) {
         _logger.info("Stopping server.");
      }
      if (_server.getServer() != null
            && _server.getServer().getState() != LifecycleState.DESTROYED) {
         if (_server.getServer().getState() != LifecycleState.STOPPED) {
            _server.getServer().stop();
         }
         _server.getServer().destroy();
      }
   }

   @Override
   public void join() throws Exception {
      _server.getServer().await();
   }

   /**
    * Returns the Tomcat server implementation.
    * Users can tune the Tomcat server for their needs if
    * the default configuration is not applicable.
    *
    * @return the underlying Tomcat server implementation
    */
   public Tomcat getServer() {
      return _server;
   }

   public Context getContext() {
      return _context;
   }

   @Override
   protected Service prepareDefaultServlet(StaticContentService staticSrv) {
      Map<String, String> initParams = new HashMap<String, String>();
      // save contentBasePath from StaticContentService to be used later
      // this path is used as base path when adding the server context to
      // the server. Only one base path is allowed per context
      // As we only create one context in Tomcat, this means that only one
      // StaticContentService can be registered (this restriction does not apply
      // in JettyServer)
      File staticContentBase = new File(staticSrv.getContentBasePath());
      _staticContentBasePath = staticContentBase.getAbsolutePath();
      initParams.put("listings", String.valueOf(staticSrv.getDirListing()));

      Service converted = new ServiceImpl();
      converted.setServlet(new DefaultServlet());
      converted.setInitParameters(initParams);
      converted.setPath(staticSrv.getPath());
      converted.setTransportGuarantee(staticSrv.getTransportGuarantee());

      return converted;
   }
}