/* **********************************************************
 * Copyright 2012, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.security;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.ini4j.Ini;
import org.ini4j.InvalidFileFormatException;
import org.ini4j.Profile.Section;

import com.vmware.vapi.internal.util.Validate;

/**
 * This class is capable of loading authentication data from an ini file
 */
public final class IniFileAuthenticationConfig implements AuthenticationConfig {

    private static final String SERVICES_SECTION_NAME = "services";
    private static final String PACKAGES_SECTION_NAME = "packages";
    private static final String OPERATIONS_SECTION_NAME = "operations";
    private static final String AUTHENTICATION_SCHEME_KEY = "AuthenticationScheme";
    private static final String SESSIONFULL_TYPE_VALUE = "SessionFull";
    private static final String SCHEME_TYPE_KEY = "Type";
    private static final String SCHEME_PREFIX = "Scheme";
    private static final String LOAD_CONFIG_ERR_MSG = "Cannot load authentication config";

    private Map<String, List<AuthnScheme>> ifaceRulesTable;
    private Map<String, List<AuthnScheme>> packageRulesTable;
    private Map<String, List<AuthnScheme>> operationRulesTable;

    public IniFileAuthenticationConfig(String configFilename) {
        Validate.notEmpty(configFilename);
        loadConfig(configFilename);
    }

    @Override
    public Map<String, List<AuthnScheme>> getPackageAuthenticationRules() {
        return packageRulesTable;
    }

    @Override
    public Map<String, List<AuthnScheme>> getIFaceAuthenticationRules() {
        return ifaceRulesTable;
    }

    @Override
    public Map<String, List<AuthnScheme>> getOperationAuthenticationRules() {
        return operationRulesTable;
    }

    /**
     * Loads the authentication configuration file into a memory structure
     *
     * @param authnConfig the file path to the authentication config file.
     *        cannot be null.
     */
    private void loadConfig(String authnConfig) {
        assert authnConfig != null;

        InputStream iniStream = AuthenticationFilter.class
                .getResourceAsStream(authnConfig);
        try {
            Ini config = new Ini(iniStream);
            constructAuthnMap(config);
        } catch (InvalidFileFormatException e) {
            throw new IllegalArgumentException(
                    "Invalid authentication config file format", e);
        } catch (IOException e) {
            throw new RuntimeException(LOAD_CONFIG_ERR_MSG, e);
        } finally {
            try {
                iniStream.close();
            } catch (IOException e) {
                throw new RuntimeException(LOAD_CONFIG_ERR_MSG, e);
            }
        }
    }

    /**
     * Constructs the authentication config memory structure
     *
     * @param config cannot be null
     */
    private void constructAuthnMap(Ini config) {
        assert config != null;

        Map<String, AuthnScheme> schemes = loadSchemes(config);
        packageRulesTable = Collections.unmodifiableMap(loadConfigSection(
                config, schemes, PACKAGES_SECTION_NAME));
        ifaceRulesTable = Collections.unmodifiableMap(loadConfigSection(
                config, schemes, SERVICES_SECTION_NAME));
        operationRulesTable = Collections.unmodifiableMap(loadConfigSection(
                config, schemes, OPERATIONS_SECTION_NAME));
    }

    /**
     * @return the authentication schemes map. cannot be null.
     */
    private Map<String, AuthnScheme> loadSchemes(Ini config) {
        assert config != null;

        Map<String, AuthnScheme> schemeList = new HashMap<String, AuthnScheme>();
        for (String sectionName : config.keySet()) {
            if (sectionName.startsWith(SCHEME_PREFIX)) {
                List<String> schemes = new ArrayList<String>();
                Section section = config.get(sectionName);
                String schemeType = section.get(SCHEME_TYPE_KEY);
                Validate.notNull(schemeType);
                if (schemeType.trim().equalsIgnoreCase(SESSIONFULL_TYPE_VALUE)) {
                    schemes.add(StdSecuritySchemes.SESSION);
                }
                String scheme = section.get(AUTHENTICATION_SCHEME_KEY);
                Validate.notNull(scheme);
                if (!scheme.trim().isEmpty()) {
                    schemes.add(scheme);
                }

                schemeList.put(sectionName, new AuthnScheme(schemes));
            }
        }

        return schemeList;
    }

    /**
     * Loads a config section authentication rules listed in the authentication
     * config file
     *
     * @param config cannot be null.
     * @param schemes the predefined authentication schemes. cannot be null.
     * @return the interface authentication rules. cannot be null.
     */
    private Map<String, List<AuthnScheme>> loadConfigSection(Ini config,
            Map<String, AuthnScheme> schemes, String sectionName) {
        assert config != null && schemes != null;

        Map<String, List<AuthnScheme>> result = new HashMap<String, List<AuthnScheme>>();
        for (String currentSectionName : config.keySet()) {
            if (currentSectionName.trim().equalsIgnoreCase(sectionName)) {
                Section section = config.get(currentSectionName);
                for (String ifaceName : section.keySet()) {
                    String schemeListString = section.get(ifaceName);
                    Validate.notNull(schemeListString);
                    List<AuthnScheme> schemeList = new ArrayList<AuthnScheme>();
                    if (!schemeListString.trim().isEmpty()) {
                        parseSchemeList(schemes, schemeListString, schemeList);
                    }
                    result.put(ifaceName.trim().toLowerCase(Locale.ENGLISH), schemeList);
                }
            }
        }
        return result;
    }

    private void parseSchemeList(Map<String, AuthnScheme> schemes,
            String schemeListString, List<AuthnScheme> schemeList) {
        for (String schemeName : schemeListString.split(",")) {
            AuthnScheme scheme = schemes.get(schemeName.trim());
            if (scheme == null) {
                throw new RuntimeException(
                        "Unknown scheme name found " + schemeName);
            }
            schemeList.add(scheme);
        }
    }
}
