/* **********************************************************
 * Copyright (c) 2013-2016, 2019, 2021 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.security;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.JsonNode;
import com.vmware.vapi.internal.util.JacksonUtil;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.internal.util.io.IoUtil;

/**
 * This class is capable of loading authentication data from a JSON file
 */
public class JsonAuthenticationConfig implements AuthenticationConfig {
    private static final Logger logger = LoggerFactory.getLogger(JsonAuthenticationConfig.class);

    private static final String AUTHENTICATION_SECTION_NAME = "authentication";
    private static final String PRODUCT_SECTION_NAME = "product";
    private static final String COMPONENT_SECTION_NAME = "component";
    private static final String SERVICES_SECTION_NAME = "services";
    private static final String PACKAGES_SECTION_NAME = "packages";
    private static final String OPERATIONS_SECTION_NAME = "operations";
    private static final String AUTHENTICATION_SCHEME_KEY = "authenticationScheme";
    private static final String SESSION_AWARE_TYPE_VALUE = "SessionAware";
    private static final String SCHEME_TYPE_KEY = "type";
    private static final String SCHEMES_SECTION = "schemes";
    private static final String LOAD_CONFIG_ERR_MSG = "Cannot load authentication config";
    private static final String FIND_CONFIG_ERR_MSG = "Cannot find authentication config file %s on the classpath";

    private Map<String, List<AuthnScheme>> ifaceRulesTable;
    private Map<String, List<AuthnScheme>> packageRulesTable;
    private Map<String, List<AuthnScheme>> operationRulesTable;

    /**
     * A resource is expected to be on the classpath and to be ASCII or UTF-8
     * encoded. It will be loaded from the thread context class loader (if set).
     * If not successful the current class loader (used to load this class) will
     * be tried. If not successful again the current class loader will try to
     * load the resource relatively to this class location. Class loaders will
     * be invoked in the specified order.
     *
     * @param resourceNames the json authn config resource names. must not be
     *                      <code>null</code>.
     * @throws RuntimeException if the configuration cannot be loaded due to
     *                          file reading errors or parsing errors
     */
    public JsonAuthenticationConfig(String... resourceNames) {
        Validate.notNull(resourceNames);
        initRules(loadAndMergeConfig(resourceNames));
    }

    /**
     * @param configReaders the readers that will be used to load the file data.
     *                     must not be null. It is responsibility of the caller
     *                     to close the reader after the constructor returns.
     * @throws RuntimeException if the configuration cannot be loaded due to
     *                          file reading errors or parsing errors
     */
    public JsonAuthenticationConfig(Reader... configReaders) {
        Validate.notNull(configReaders);
        initRules(loadAndMergeConfig(configReaders));
    }

    @Override
    public Map<String, List<AuthnScheme>> getPackageAuthenticationRules() {
        return packageRulesTable;
    }

    @Override
    public Map<String, List<AuthnScheme>> getIFaceAuthenticationRules() {
        return ifaceRulesTable;
    }

    @Override
    public Map<String, List<AuthnScheme>> getOperationAuthenticationRules() {
        return operationRulesTable;
    }

    private void initRules(Rules rules) {
        this.ifaceRulesTable = Collections.unmodifiableMap(rules.ifaceTable);
        this.operationRulesTable = Collections.unmodifiableMap(rules.operationTable);
        this.packageRulesTable = Collections.unmodifiableMap(rules.packageTable);
    }

    /**
     * Loads an array of authentication configuration files
     *
     * @param configs
     * @return the merged authentication rules
     */
    private Rules loadAndMergeConfig(String[] configs) {
        Rules result = new Rules();
        for (String config : configs) {
            Validate.notNull(config);
            Rules rules = loadConfig(config);
            result.mergeConfigs(rules);
        }

        return result;
    }

    /**
     * Loads an array of authentication configuration files
     *
     * @param configs
     * @return the merged authentication rules
     */
    private Rules loadAndMergeConfig(Reader[] configs) {
        Rules result = new Rules();
        for (Reader config : configs) {
            Validate.notNull(config);
            Rules rules = loadConfig(config);
            result.mergeConfigs(rules);
        }

        return result;
    }


    /**
     * Loads the authentication configuration file into a memory structure
     *
     * @param configReader the reader that will be used for loading the file data.
     *        must not be null.
     */
    private Rules loadConfig(Reader configReader) {
        try {
            JsonNode rootNode = JacksonUtil.readObjectFieldValue(configReader, AUTHENTICATION_SECTION_NAME);
            return constructAuthnMap(rootNode);
        } catch (IOException e) {
            throw new RuntimeException(LOAD_CONFIG_ERR_MSG, e);
        }
    }

    /**
     * Loads the authentication configuration file into a memory structure
     *
     * @param authnConfig the file path to the authentication config file.
     *        cannot be null.
     */
    private Rules loadConfig(String authnConfig) {
        InputStreamReader configStream = new InputStreamReader(
                getInputStream(authnConfig));
        try {
            return loadConfig(configStream);
        } finally {
            IoUtil.silentClose(configStream);
        }
    }

    /**
     * Constructs the authentication config memory structure.
     */
    private Rules constructAuthnMap(JsonNode authnNode) {
        assert authnNode != null;

        JsonNode componentNode = authnNode.get(COMPONENT_SECTION_NAME);
        if (componentNode == null) {
            // backwards compatibility for metadata-generator JSON
            componentNode = authnNode.get(PRODUCT_SECTION_NAME);
        }

        assert componentNode != null;

        Map<String, AuthnScheme> schemes = loadSchemes(componentNode);
        Rules rules = new Rules();
        rules.packageTable = loadConfigSection(componentNode.get(PACKAGES_SECTION_NAME), schemes);
        rules.ifaceTable = loadConfigSection(componentNode.get(SERVICES_SECTION_NAME), schemes);
        rules.operationTable = loadConfigSection(componentNode.get(OPERATIONS_SECTION_NAME),
                                                 schemes);
        return rules;
    }

    /**
     * @return the authentication schemes map. cannot be null.
     */
    private Map<String, AuthnScheme> loadSchemes(JsonNode componentNode) {
        Map<String, AuthnScheme> result = new HashMap<>();
        JsonNode schemesNode = componentNode.get(SCHEMES_SECTION);
        if (schemesNode == null) {
            logger.debug("No 'schemes' node found");
            return result;
        }
        Iterator<Entry<String, JsonNode>> schemesIterator = schemesNode.fields();
        while (schemesIterator.hasNext()) {
            Entry<String, JsonNode> scheme = schemesIterator.next();
            logger.debug("Parsing authentication scheme: {}", scheme.getKey());
            result.put(scheme.getKey(), createAuthnScheme(scheme.getValue()));
        }

        return result;
    }

    /**
     * Parses a JSON node that contains all fields required for creating an
     * {@link AuthnScheme}
     *
     * @param node cannot be null
     * @return the {@link AuthnScheme} that is a result from parsing this node
     */
    private AuthnScheme createAuthnScheme(JsonNode node) {
        assert node != null;

        List<String> schemes = new ArrayList<>();

        JsonNode type = node.findValue(SCHEME_TYPE_KEY);
        Validate.notNull(type);
        if (type.textValue().equalsIgnoreCase(SESSION_AWARE_TYPE_VALUE)) {
            schemes.add(StdSecuritySchemes.SESSION);
        }

        JsonNode scheme = node.get(AUTHENTICATION_SCHEME_KEY);
        Validate.notNull(scheme);
        if (!scheme.textValue().trim().isEmpty()) {
            schemes.add(scheme.textValue());
        }

        return new AuthnScheme(schemes);
    }

    /**
     * Loads a config section authentication rules listed in the authentication
     * config file
     *
     * @param rootNode cannot be null.
     * @param schemes the predefined authentication schemes. cannot be null.
     * @return the interface authentication rules. cannot be null.
     */
    private Map<String, List<AuthnScheme>> loadConfigSection(JsonNode rootNode,
            Map<String, AuthnScheme> schemes) {
        assert rootNode != null && schemes != null;

        Map<String, List<AuthnScheme>> result = new HashMap<>();
        Iterator<Entry<String, JsonNode>> rules = rootNode.fields();
        while (rules.hasNext()) {
            Entry<String, JsonNode> rule = rules.next();
            JsonNode schemeListNode = rule.getValue();
            Validate.isTrue(schemeListNode.isArray());
            List<AuthnScheme> schemeList = new ArrayList<>();
            for (JsonNode schemeName : schemeListNode) {
                AuthnScheme scheme = schemes.get(schemeName.asText());
                if (scheme == null) {
                    throw new RuntimeException(
                            "Unknown scheme name found " + schemeName);
                }
                schemeList.add(scheme);
            }
            result.put(rule.getKey(), schemeList);
        }

        return result;
    }

    /**
     * This method tries to get an input stream which could load the data from
     * the resource. The method first tries loading the resource from the thread
     * context class loader and then falls back to the default class loader.
     *
     * @param resourceName cannot be null
     * @return the input stream that could be used for loading the resource data
     * @throws RuntimeException if the resource cannot be loaded
     */
    private InputStream getInputStream(String resourceName) {
        InputStream result = null;

        // try loading the file using the context class loader
        result = getInputStream(resourceName, Thread.currentThread().getContextClassLoader());
        if (result != null) {
            return result;
        }

        // try loading the file using the default class loader
        result = getInputStream(resourceName, this.getClass().getClassLoader());
        if (result != null) {
            return result;
        }

        // try loading the file if the filename starts with '/' i.e. relatively
        // to JsonAuthenticationConfig class path using the default class loader
        // to keep backwards compatibility
        result = this.getClass().getResourceAsStream(resourceName);
        if (result != null) {
            return result;
        }

        // resource cannot be loaded
        throw new RuntimeException(String.format(FIND_CONFIG_ERR_MSG, resourceName));
    }

    /**
     * @param resourceName must not be null
     * @param cl           can be null
     * @return the resource stream if the cl succeed to load it;
     *         otherwise <code>null</code>
     */
    private InputStream getInputStream(String resourceName, ClassLoader cl) {
        return (cl != null)? cl.getResourceAsStream(resourceName) : null;
    }

    /**
     * Contains the authentication rules
     */
    private final class Rules {
        private Map<String, List<AuthnScheme>> ifaceTable = new HashMap<>();
        private Map<String, List<AuthnScheme>> packageTable = new HashMap<>();
        private Map<String, List<AuthnScheme>> operationTable = new HashMap<>();

        /**
         * @param rules the new rules that will be merged with the rules from
         *              this instance. must not be <code>null</code>
         */
        public void mergeConfigs(Rules rules) {
            // merge interfaces
            mergeRules(ifaceTable, rules.ifaceTable);
            // merge operations
            mergeRules(operationTable, rules.operationTable);
            // merge packages
            mergeRules(packageTable, rules.packageTable);
        }

        /**
         * Merges newRules over the baseRules
         *
         * @param baseRules
         * @param newRules
         * @throws RuntimeException if there is a merge conflict
         */
        private void mergeRules(Map<String, List<AuthnScheme>> baseRules,
                                Map<String, List<AuthnScheme>> newRules) {
            for (String entity : newRules.keySet()) {
                if (baseRules.containsKey(entity) &&
                    !baseRules.get(entity).equals(newRules.get(entity))) {
                    String mergeErrMsg = String.format(
                                "Merge conflict for entity %s", entity);
                    logger.warn(mergeErrMsg);
                    throw new RuntimeException(mergeErrMsg);
                }
                baseRules.put(entity, newRules.get(entity));
            }
        }
    }
}
