/* **********************************************************
 * Copyright 2012, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

/*
 * SessionSecurityContext.java --
 *
 *      SecurityContext for request authentication with a session ID
 */
package com.vmware.vapi.security;

import java.util.Arrays;
import java.util.Collections;
import java.util.Map;

import com.vmware.vapi.core.ExecutionContext.SecurityContext;
import com.vmware.vapi.internal.util.Validate;

/**
 * This class represents the security context needed for authentication
 * a session ID.
 */
public final class SessionSecurityContext implements SecurityContext {
    public static final String SESSION_ID_KEY = "sessionId";

    private final char[] sessionId;

    /**
     *
     * @param sessionId
     * @throws IllegalArgumentException if <code>sessionId</code> is
     *                                  <code>null</code> or empty
     */
    public SessionSecurityContext(char[] sessionId) {
        Validate.notNull(sessionId, "'sessionId' must not be null");
        Validate.isTrue(sessionId.length > 0, "'sessionId' must not be empty");

        this.sessionId = Arrays.copyOf(sessionId, sessionId.length);
    }

    @Override
    public Object getProperty(String key) {
        if (SESSION_ID_KEY.equals(key)) {
            return sessionId;
        } else if (SecurityContext.AUTHENTICATION_SCHEME_ID.equals(key)) {
            return StdSecuritySchemes.SESSION;
        }

        return null;
    }

    @Override
    public Map<String, Object> getAllProperties() {
        return Collections.<String, Object>singletonMap(
                SESSION_ID_KEY, sessionId);
    }

    public char[] getSessionId() {
        return Arrays.copyOf(sessionId, sessionId.length);
    }

    /**
     * Creates a {@link SessionSecurityContext} from the given
     * {@link SecurityContext} if possible. Otherwise <code>null</code>
     * will be returned.
     *
     * @param context cannot be null.
     * @return the parsed {@link SessionSecurityContext}.
     */
    public static SessionSecurityContext newInstance(SecurityContext context) {
        Validate.notNull(context);
        String sessionIdStr = getAsString(context.getProperty(SESSION_ID_KEY));
        char[] sessionId = (sessionIdStr == null)?
                getAsCharArr(context.getProperty(SESSION_ID_KEY)) : sessionIdStr.toCharArray();
        if (sessionId != null) {
            return new SessionSecurityContext(sessionId);
        }

        return null;
    }

    /**
     * @param value can be null.
     * @return the given value as a String if possible. otherwise
     * <code>null</code> will be returned.
     */
    private static String getAsString(Object value) {
        if (value instanceof String) {
            return (String) value;
        }

        return null;
    }

    /**
     * @param value can be null.
     * @return the given value as a String if possible. otherwise
     * <code>null</code> will be returned.
     */
    private static char[] getAsCharArr(Object value) {
        if (value instanceof char[]) {
            return (char[]) value;
        }

        return null;
    }
}
