/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.std;

import com.vmware.vapi.internal.util.Validate;

/**
 * Progress notification which is emitted by a running operation.
 * <p>
 * Progress can be either determinate or indeterminate. Determinate progress
 * gives an estimate of how much work is already done and how much more work
 * remains. Indeterminate progress means that progress is being made but it is
 * not clear how much more work remains. For determinate progress
 * {@link #getMin()} is strictly less than {@link #getMax()} and
 * {@link #getCurrent()} is a value in the closed interval from
 * {@link #getMin()} to {@link #getMax()}. For indeterminate progress
 * {@link #getMin()} must be equal to {@link #getMax()} and
 * {@link #getCurrent()} is any non-negative value.
 * <p>
 * The same determinate progress can be represented in multiple ways (all
 * semantically equivalent). For example 50% progress can be represented as
 * (min=0, max=100, current=50) or (min=0, max=10, current=5) or (min=0, max=4,
 * current=2) etc.
 * <p>
 * The order of progress notifications is not guaranteed. The consumer of
 * progress notifications should ignore progress notifications that go backwards
 * (i.e. decreasing current progress value).
 */
public final class Progress {

    private final long min;
    private final long max;
    private final long current;

    /**
     * Creates a progress notification.
     *
     * @param min minimum value in the progress range; must be non-negative
     * @param max maximum value in the progress range; must be non-negative
     * @param current current progress value; must be non-negative
     * @throws IllegalArgumentException if the field values are invalid
     */
    public Progress(long min, long max, long current) {
        Validate.isTrue(min >= 0, "'min' must be non-negative");
        Validate.isTrue(max >= 0, "'max' must be non-negative");
        Validate.isTrue(current >= 0, "'current' must be non-negative");
        Validate.isTrue(min <= max, "'min' must not be greater than 'max'");
        this.min = min;
        this.max = max;
        this.current = current;
    }

    /**
     * @return minimum value in the progress range
     */
    public long getMin() {
        return min;
    }

    /**
     * @return maximum value in the progress range
     */
    public long getMax() {
        return max;
    }

    /**
     * @return current progress value
     */
    public long getCurrent() {
        return current;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (!(obj instanceof Progress)) {
            return false;
        }
        Progress other = (Progress) obj;
        return this.min == other.min
                && this.max == other.max
                && this.current == other.current;
    }

    @Override
    public int hashCode() {
        long hash = 11;
        hash = hash * 31 + min;
        hash = hash * 31 + max;
        hash = hash * 31 + current;
        return (int) hash;
    }

    @Override
    public String toString() {
        StringBuilder s = new StringBuilder();
        s.append("progress: {");
        s.append("min=");
        s.append(min);
        s.append(", max=");
        s.append(max);
        s.append(", current=");
        s.append(current);
        s.append("}");
        return s.toString();
    }
}
