/* **********************************************************
 * Copyright (c) 2012-2014, 2016, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.std;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.Message;
import com.vmware.vapi.data.DataDefinition;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.DoubleDefinition;
import com.vmware.vapi.data.DynamicStructDefinition;
import com.vmware.vapi.data.ErrorDefinition;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.data.IntegerDefinition;
import com.vmware.vapi.data.ListDefinition;
import com.vmware.vapi.data.ListValue;
import com.vmware.vapi.data.OptionalDefinition;
import com.vmware.vapi.data.OptionalValue;
import com.vmware.vapi.data.StringDefinition;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.data.StructDefinition;
import com.vmware.vapi.data.StructRefDefinition;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.util.Validate;

/**
 * Factory for {@link DataDefinition}s and {@link DataValue}s for standard vAPI
 * structures, like errors and localizable messages.
 */
public class StandardDataFactory {
    public static final String VALUE_FIELD_NAME = "value";
    public static final String KEY_FIELD_NAME = "key";
    public static final String MAP_ENTRY_TYPE = "map-entry";
    private static final StringDefinition STRING_DEF =
            StringDefinition.getInstance();
    private static final OptionalDefinition OPT_STRING_DEF =
            new OptionalDefinition(STRING_DEF);
    private static final OptionalDefinition OPT_LONG_DEF =
            new OptionalDefinition(IntegerDefinition.getInstance());
    private static final OptionalDefinition OPT_DOUBLE_DEF =
            new OptionalDefinition(DoubleDefinition.getInstance());
    // names for standard errors
    public static final String ALREADY_EXISTS =
            "com.vmware.vapi.std.errors.already_exists";
    public static final String INTERNAL_SERVER_ERROR =
            "com.vmware.vapi.std.errors.internal_server_error";
    public static final String OPERATION_NOT_FOUND =
            "com.vmware.vapi.std.errors.operation_not_found";
    public static final String INVALID_ARGUMENT =
            "com.vmware.vapi.std.errors.invalid_argument";
    public static final String NOT_FOUND =
            "com.vmware.vapi.std.errors.not_found";
    public static final String UNAUTHENTICATED =
            "com.vmware.vapi.std.errors.unauthenticated";
    public static final String UNEXPECTED_INPUT =
            "com.vmware.vapi.std.errors.unexpected_input";

    // name for localizable message structure
    public static final String MESSAGE_STRUCT_NAME =
            "com.vmware.vapi.std.localizable_message";
    public static final String PARAM_STRUCT_NAME =
            "com.vmware.vapi.std.localization_param";
    public static final String NESTED_MSG_STRUCT_NAME =
            "com.vmware.vapi.std.nested_localizable_message";

    // LocalizableMessage fields
    public static final String ID_FIELD_NAME = "id";
    public static final String DEFAUL_MSG_FIELD_NAME = "default_message";
    public static final String ARGS_FIELD_NAME = "args";
    public static final String LOCALIZED_FIELD_NAME = "localized";
    public static final String PARAMS_FIELD_NAME = "params";

    // std error fields
    public static final String MESSAGES_FIELD_NAME = "messages";
    public static final String DATA_FIELD_NAME = "data";
    public static final String TYPE_FIELD_NAME = "error_type";

    private static StructDefinition locMessageDef;


    private static DataDefinition mapDef(DataDefinition key,
                                         DataDefinition value) {
        Map<String, DataDefinition> fields = new HashMap<>();
        fields.put(KEY_FIELD_NAME, key);
        fields.put(VALUE_FIELD_NAME, value);
        return new ListDefinition(new StructDefinition(MAP_ENTRY_TYPE, fields));
    }

    private static DataDefinition nestedDef() {
        StructRefDefinition ref = new StructRefDefinition(PARAM_STRUCT_NAME);
        Map<String, DataDefinition> fields = new HashMap<>();
        fields.put(ID_FIELD_NAME, STRING_DEF);
        fields.put(PARAMS_FIELD_NAME, new OptionalDefinition(mapDef(STRING_DEF, ref)));
        return new StructDefinition(NESTED_MSG_STRUCT_NAME, fields);
    }

    private static DataDefinition paramDef() {
        Map<String, DataDefinition> fields = new HashMap<>();
        fields.put("s", OPT_STRING_DEF);
        fields.put("dt", OPT_STRING_DEF);
        fields.put("d", OPT_DOUBLE_DEF);
        fields.put("precision", OPT_LONG_DEF);
        fields.put("format", OPT_STRING_DEF);
        fields.put("i", OPT_LONG_DEF);
        fields.put("l", new OptionalDefinition(nestedDef()));
        return new StructDefinition(PARAM_STRUCT_NAME, fields);
    }

    static {
        Map<String, DataDefinition> msgFields = new HashMap<>();
        msgFields.put(ID_FIELD_NAME, STRING_DEF);
        msgFields.put(DEFAUL_MSG_FIELD_NAME, STRING_DEF);
        msgFields.put(ARGS_FIELD_NAME,
                      new ListDefinition(STRING_DEF));
        msgFields.put(LOCALIZED_FIELD_NAME, OPT_STRING_DEF);
        msgFields.put(PARAMS_FIELD_NAME,
                      new OptionalDefinition(mapDef(STRING_DEF, paramDef())));

        locMessageDef = new StructDefinition(MESSAGE_STRUCT_NAME, msgFields);
    }

    public static final Map<String, DataDefinition> STANDARD_ERROR_FIELDS;
    static {
        STANDARD_ERROR_FIELDS = new HashMap<>();
        STANDARD_ERROR_FIELDS.put(MESSAGES_FIELD_NAME,
                                  new ListDefinition(locMessageDef));
        STANDARD_ERROR_FIELDS.put(DATA_FIELD_NAME,
                                  new OptionalDefinition(DynamicStructDefinition
                                          .getInstance()));
        STANDARD_ERROR_FIELDS.put(TYPE_FIELD_NAME, OPT_STRING_DEF);
    }

    public static StructDefinition getLocalizableMessageDefinition() {
        return locMessageDef;
    }

    /**
     * Creates {@link ErrorDefinition}s for standard vAPI errors. This includes
     * definitions for nested data types such as {@link LocalizableMessage},
     * {@link NestedLocalizableMessage} and {@link LocalizationParam} too.
     *
     * @param errorName the error name
     * @return new {@link ErrorDefinition} instance
     */
    public static ErrorDefinition createStandardErrorDefinition(String errorName) {

        return new ErrorDefinition(errorName, STANDARD_ERROR_FIELDS);
    }

    /**
     * Creates a standard error <code>ErrorValue</code> instance for the given
     * name list of {@link Message}s.
     *
     * @see #createErrorValueForMessages(ErrorDefinition, List)
     */
    public static ErrorValue createErrorValueForMessages(String errorName,
                                                         List<Message> messages) {

        return createErrorValueForMessages(
                createStandardErrorDefinition(errorName),
                messages);
    }

    /**
     * Creates instance for given error definition and list of {@link Message}s.
     *
     * @param errorDef error definition
     * @param messages the messages for the new {@link ErrorValue}
     * @return new {@link ErrorValue} instance
     *
     * @throws IllegalArgumentException if {@code messages} or {@code errorDef}
     *         is {@code null} or {@code messages} is empty or contains
     *         {@code null} element(s)
     */
    public static ErrorValue createErrorValueForMessages(ErrorDefinition errorDef,
                                                         List<Message> messages) {
        Validate.notNull(errorDef);
        Validate.notNull(messages);

        if (messages.size() < 1) {
            throw new IllegalArgumentException("The messages parameter must"
                    + " contain at least 1 message");
        }

        try {
            if (messages.contains(null)) {
                throw new IllegalArgumentException("null message detected");
            }
        } catch (NullPointerException ex) {
            // List impl doesn't support null at all; so it's fine
            // it can't contain null message
        }

        ListValue msgListValue = new ListValue();
        for (Message m : messages) {
            msgListValue.add(convertMessageToDataValue(m));
        }
        String discriminator = extractDiscriminator(errorDef.getName());

        ErrorValue result = errorDef.newInstance();
        result.setField(MESSAGES_FIELD_NAME, msgListValue);
        result.setField(DATA_FIELD_NAME, OptionalValue.UNSET);
        result.setField(TYPE_FIELD_NAME, new OptionalValue(discriminator));
        return result;
    }

    static String extractDiscriminator(String name) {
        String discriminator = name;
        int lastDotIndex = discriminator.lastIndexOf('.');
        if (lastDotIndex > -1) {
            discriminator = discriminator.substring(lastDotIndex + 1);
        }
        return discriminator.toUpperCase(Locale.ENGLISH);
    }

    private static StructValue convertMessageToDataValue(Message msg) {
        StructDefinition resultDef = getLocalizableMessageDefinition();
        StructValue result = resultDef.newInstance();

        result.setField(ID_FIELD_NAME, msg.getId());
        result.setField(DEFAUL_MSG_FIELD_NAME, msg.getDefaultMessage());
        result.setField(ARGS_FIELD_NAME, convertArgsToDataValue(msg.getArgs()));
        result.setField(LOCALIZED_FIELD_NAME, OptionalValue.UNSET);
        result.setField(PARAMS_FIELD_NAME, OptionalValue.UNSET);

        return result;
    }

    private static DataValue convertArgsToDataValue(String[] args) {
        ListValue result = new ListValue();

        if (args != null) {
            for (String argument : args) {
                result.add(new StringValue(argument));
            }
        }

        return result;
    }

    /**
     * Returns messages associated with the specified standard error
     * {@link ErrorValue} represented as list of {@link Message}s. This
     * method assumes the structure of "messages" field from
     * {@link #createErrorValueForMessages(ErrorDefinition, List)}. In case this
     * structure is not present - empty list is returned.
     *
     * @param error standard error value
     * @return list of {@link Message} instances
     */
    public static List<Message> getMessagesFromErrorValue(StructValue error) {
        try {
            ListValue messageList = error.getList(MESSAGES_FIELD_NAME);
            List<Message> result = new ArrayList<>();
            for (DataValue messageStruct : messageList) {
                result.add(buildLegacyMessage((StructValue) messageStruct));
            }
            return result;
        } catch (CoreException ex) {
            // "messages" field doesn't have the expected structure - can't
            // extract any messages
            return Collections.emptyList();
        }
    }

    private static Message buildLegacyMessage(StructValue messageStruct) {
        String id = messageStruct.getString(ID_FIELD_NAME);
        String defaultMessage = messageStruct.getString(DEFAUL_MSG_FIELD_NAME);
        ListValue argsList = messageStruct.getList(ARGS_FIELD_NAME);
        List<String> stringArgsList = argsList.getStringList();

        return new Message(id,
                           defaultMessage,
                           stringArgsList
                                   .toArray(new String[stringArgsList.size()]));
    }
}
