/* **********************************************************
 * Copyright 2012-2014, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.std.activation.impl;

import static com.vmware.vapi.MessageFactory.getMessage;

import java.util.Collections;
import java.util.Set;

import com.vmware.vapi.activation.Activations;
import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.DecoratorApiProvider;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.InterfaceIdentifier;
import com.vmware.vapi.core.MethodIdentifier;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ErrorDefinition;
import com.vmware.vapi.internal.std.activation.impl.ActivationRegistry;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.provider.introspection.ErrorAugmentingFilter;
import com.vmware.vapi.std.StandardDataFactory;
import com.vmware.vapi.util.async.DecoratorAsyncHandle;

/**
 * <b>WARNING:</b> Use only as a sample. The API is experimental and subject to
 * change in future versions.
 *
 * <p>
 * Server-side filter which intercepts method invocations and registers an
 * activation identifier for each invocation into the {@link AcitvationRegistry}
 * . As soon as the method invocation completes, the activation gets
 * unregistered. If a duplication of identifiers occurs, the filter reports the
 * {@link StandardDataFactory#ALREADY_EXISTS} standard error. The filter does not register
 * activations for the methods of the activation manager.
 * </p>
 * <p>
 * This filter implementation is specific to the in-memory activation registry
 * implementation in {@link ActivationManagerImpl}. If you write your own
 * activation registry, you should also write your own activation filter.
 * </p>
 */
public final class ActivationFilter extends DecoratorApiProvider {

    /**
     * Interface identifier of the activation manager service.
     */
    public static final InterfaceIdentifier ACTIVATION_MANAGER_ID = new InterfaceIdentifier(
            "com.vmware.vapi.std.activation.activation_manager");

    private static final Set<ErrorDefinition> ACTIVATION_FILTER_ERROR_DEFS =
            Collections.singleton(
                    StandardDataFactory.createStandardErrorDefinition(
                            StandardDataFactory.ALREADY_EXISTS));

    private final ActivationRegistry activationRegistry;

    /**
     * Creates an activation filter.
     *
     * @param decoratedProvider next provider in the stack; must not be
     *                          <code>null</code>
     * @param activationRegistry in-memory activation registry; must not be
     *                          <code>null</code>
     * @throws IllegalArgumentException if any of the arguments is
     *                                  <code>null</code>
     */
    public ActivationFilter(ApiProvider decoratedProvider,
                            ActivationRegistry activationRegistry) {
        super(new ErrorAugmentingFilter(decoratedProvider,
                ACTIVATION_FILTER_ERROR_DEFS));
        Validate.notNull(activationRegistry);
        this.activationRegistry = activationRegistry;
    }

    @Override
    public void invoke(String serviceId,
                       String operationId,
                       DataValue input,
                       ExecutionContext ctx,
                       AsyncHandle<MethodResult> asyncHandle) {

        MethodIdentifier method = new MethodIdentifier(
                new InterfaceIdentifier(serviceId),
                operationId);

        // do not register activations for the methods of the activation manager
        // will not be able to cancel a cancel() method invocation
        if (ACTIVATION_MANAGER_ID.equals(method.getInterfaceIdentifier())) {
            decoratedProvider.invoke(serviceId, operationId, input, ctx, asyncHandle);
            return;
        }

        String activationIdFromClient = Activations.getActivationId(ctx);

        final ExecutionContext delegatedCtx;
        final String activationId;
        if (activationIdFromClient == null) {
            // the client has not provided activation identifier -> generate
            activationId = Activations.newActivationId();
            delegatedCtx = Activations.setActivationId(ctx, activationId);
        } else {
            activationId = activationIdFromClient;
            delegatedCtx = ctx;
        }

        try {
            activationRegistry.register(activationId, method);
        } catch (IllegalStateException ex) {
            // duplicated activation identifier
            asyncHandle.setResult(MethodResult
                    .newErrorResult(StandardDataFactory
                            .createErrorValueForMessages(
                                    StandardDataFactory.ALREADY_EXISTS,
                                    Collections.singletonList(getMessage(
                                            "vapi.activation.id.duplicate",
                                            activationId)))));
            // TODO: it is a security problem that activation ids can collide;
            // ideally activations should be scoped by owner user principal
            return;
        }
        // clean up the activation after the invocation is complete
        AsyncHandle<MethodResult> cb;
        cb = new DecoratorAsyncHandle<MethodResult>(asyncHandle) {
            @Override
            public void setResult(MethodResult result) {
                activationRegistry.unregister(activationId);
                super.setResult(result);
            }
            @Override
            public void setError(RuntimeException error) {
                activationRegistry.unregister(activationId);
                super.setError(error);
            }
        };
        decoratedProvider.invoke(serviceId, operationId, input, delegatedCtx, cb);
    }
}
