/* **********************************************************
 * Copyright 2012-2013, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.std.activation.impl;

import java.util.Arrays;
import java.util.Collections;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.vmware.vapi.Message;
import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.activation.ActivationMonitor;
import com.vmware.vapi.bindings.server.AsyncContext;
import com.vmware.vapi.core.MethodIdentifier;
import com.vmware.vapi.internal.std.activation.impl.ActivationRegistry;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.std.LocalizableMessage;
import com.vmware.vapi.std.activation.ActivationManagerProvider;
import com.vmware.vapi.std.errors.NotFound;

/**
 * <b>WARNING:</b> Use only as a sample. The API is experimental and subject to
 * change in future versions.
 *
 * <p>
 * In memory registry for activations. It is both an implementation of an
 * activation manager service (used by clients to cancel method invocations) and
 * a {@link ActivationMonitor} (used by service implementations to check for
 * their own cancellation status). It requires an {@link ActivationFilter} to be
 * deployed.
 * </p>
 *
 * <p>
 * <i>Thread-safety:</i> the implementation is thread-safe.
 * </p>
 */
public class ActivationManagerImpl extends ActivationMonitor implements
        ActivationManagerProvider, ActivationRegistry {

    /**
     * Registry entry about a single activation.
     * <p>
     * <i>Thread-safety:</i> the implementation is thread-safe.
     * </p>
     */
    private static final class ActivationEntry {

        private volatile boolean canceled = false;

        /**
         * Checks whether the activation is requested to be canceled.
         *
         * @return whether the activation is requested to be canceled
         */
        public boolean isCanceled() {
            return canceled;
        }

        /**
         * Marks the activation as requested to be canceled.
         */
        public void cancel() {
            canceled = true;
        }
    }

    /**
     * In-memory collection of entries for all running activations.
     */
    private final ConcurrentMap<String, ActivationEntry> activations =
            new ConcurrentHashMap<String, ActivationEntry>();

    // interface ActivationRegistry
    @Override
    public void register(String activationId, MethodIdentifier methodId)
            throws IllegalArgumentException, IllegalStateException {
        Validate.notNull(activationId);
        Validate.notNull(methodId);
        ActivationEntry entry = new ActivationEntry();
        ActivationEntry old = activations.putIfAbsent(activationId, entry);
        if (old != null) {
            throw new IllegalStateException();
        }
    }

    // interface ActivationRegistry
    @Override
    public void unregister(String activationId)
            throws IllegalArgumentException {
        Validate.notNull(activationId);
        activations.remove(activationId);
    }

    // interface ActivationMonitor
    @Override
    public boolean isCancelRequested(String activationId) {
        if (activationId == null) {
            return false;
        }
        ActivationEntry entry = activations.get(activationId);
        if (entry == null) {
            return false;
        }
        return entry.isCanceled();
    }

    @Override
    public void cancel(String activationId, AsyncContext<Void> asyncContext) {
        ActivationEntry entry = getEntry(activationId);
        entry.cancel();
        asyncContext.setResult(null);
    }

    /**
     * Finds registry entry about the specified activation.
     *
     * @param activationId activation identifier; must not be <code>null</code>
     * @return entry from the registry about the activation
     * @throws NotFound there is no entry for the specified identifier in the
     *                  registry
     */
    private ActivationEntry getEntry(String activationId) throws NotFound {
        Validate.notNull(activationId);
        ActivationEntry entry = activations.get(activationId);
        // TODO: move messages for this provider outside the runtime's
        //       ResourceBundle/MessageFactory, then use
        //       LocalizableMessageFactory
        Message msg = MessageFactory.getMessage("vapi.activation.id.not.found",
                                                activationId);
        LocalizableMessage locMsg = new LocalizableMessage(msg.getId(),
                                                           msg.getDefaultMessage(),
                                                           Arrays.asList(msg.getArgs()));
        if (entry == null) {
            throw new NotFound(Collections.singletonList(locMsg), null);
        }
        return entry;
    }

}
