/*
 * Decompiled with CFR 0.152.
 */
package com.vmware.cis.data.internal.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class TaskExecutor {
    private static final Logger _logger = LoggerFactory.getLogger(TaskExecutor.class);
    private static final String ERROR_EXECUTION_CANCELLED = "Data Service execution was cancelled.";
    private final ExecutorService _executor;
    private final ErrorHandlingPolicy _errorHandling;

    public TaskExecutor(ExecutorService executor, ErrorHandlingPolicy errorHandling) {
        this._executor = executor;
        this._errorHandling = errorHandling;
    }

    public <T> List<T> invokeTasks(List<Callable<T>> tasks) {
        if (tasks.size() == 0) {
            return Collections.emptyList();
        }
        List<Future<T>> futures = this.submitForExecution(tasks);
        assert (tasks.size() == futures.size());
        ArrayList<T> results = new ArrayList<T>(tasks.size());
        ArrayList<Throwable> failedTaskErrors = new ArrayList<Throwable>();
        for (int i = 0; i < tasks.size(); ++i) {
            T result;
            Callable<T> task = tasks.get(i);
            Future<T> future = futures.get(i);
            try {
                result = future.get();
            }
            catch (InterruptedException e) {
                TaskExecutor.cancelAll(futures);
                Thread.currentThread().interrupt();
                throw new RuntimeException(ERROR_EXECUTION_CANCELLED, e);
            }
            catch (ExecutionException e) {
                Throwable actualError = TaskExecutor.unwrapExecutionException(e);
                this.applyErrorHandlingPolicy(task, actualError, futures);
                failedTaskErrors.add(actualError);
                _logger.warn("Data Service execution for task: {} has failed: {}", new Object[]{task, e.getMessage(), actualError});
                continue;
            }
            results.add(result);
        }
        assert (this._errorHandling != ErrorHandlingPolicy.STRICT || failedTaskErrors.isEmpty());
        RuntimeException error = TaskExecutor.toAggregatedError(failedTaskErrors);
        if (error != null && results.isEmpty()) {
            throw error;
        }
        return results;
    }

    private <T> List<Future<T>> submitForExecution(List<Callable<T>> tasks) {
        ArrayList<Future<T>> futures = new ArrayList<Future<T>>(tasks.size());
        for (int i = 0; i < tasks.size(); ++i) {
            if (Thread.currentThread().isInterrupted()) {
                throw new RuntimeException(ERROR_EXECUTION_CANCELLED);
            }
            Callable<T> task = tasks.get(i);
            if (this._executor == null || i == tasks.size() - 1) {
                this.executeDirectly(task, futures);
                continue;
            }
            try {
                Future<T> future = this._executor.submit(task);
                futures.add(future);
                continue;
            }
            catch (RejectedExecutionException e) {
                _logger.debug("Task {} was rejected from the thread pool and will be run in the caller thread.", task, (Object)e);
                this.executeDirectly(task, futures);
            }
        }
        return futures;
    }

    private <T> void executeDirectly(Callable<T> task, List<Future<T>> taskFutures) {
        Object result = null;
        Exception error = null;
        try {
            result = task.call();
        }
        catch (Exception e) {
            error = e;
        }
        assert (result == null || error == null);
        this.applyErrorHandlingPolicy(task, error, taskFutures);
        taskFutures.add(new TaskResultFutureAdapter<Object>(result, error));
    }

    private <T> void applyErrorHandlingPolicy(Callable<T> task, Throwable error, List<Future<T>> runningTasks) {
        if (error == null || this._errorHandling == ErrorHandlingPolicy.LENIENT) {
            return;
        }
        TaskExecutor.cancelAll(runningTasks);
        if (error instanceof RuntimeException) {
            throw (RuntimeException)error;
        }
        if (error instanceof Error) {
            throw (Error)error;
        }
        throw new RuntimeException("Execution error while running Data Service task: " + task, error);
    }

    private static <T> void cancelAll(List<Future<T>> taskFutures) {
        for (Future<T> future : taskFutures) {
            future.cancel(true);
        }
    }

    private static RuntimeException toAggregatedError(List<Throwable> errors) {
        if (errors.isEmpty()) {
            return null;
        }
        if (errors.size() == 1) {
            return TaskExecutor.toRuntimeException(errors.iterator().next());
        }
        RuntimeException aggregated = TaskExecutor.findIllegalArgumentException(errors);
        if (aggregated == null) {
            aggregated = new RuntimeException("Data Service execution error.");
        }
        for (Throwable error : errors) {
            if (aggregated == error) continue;
            aggregated.addSuppressed(error);
        }
        return aggregated;
    }

    private static Throwable unwrapExecutionException(Exception e) {
        if (e instanceof ExecutionException) {
            return e.getCause();
        }
        return e;
    }

    private static RuntimeException toRuntimeException(Throwable error) {
        if (error instanceof RuntimeException) {
            return (RuntimeException)error;
        }
        return new RuntimeException(error.getMessage(), error);
    }

    private static IllegalArgumentException findIllegalArgumentException(List<Throwable> errors) {
        for (Throwable error : errors) {
            if (!(error instanceof IllegalArgumentException)) continue;
            return (IllegalArgumentException)error;
        }
        return null;
    }

    private static final class TaskResultFutureAdapter<T>
    implements Future<T> {
        private final T _result;
        private final Throwable _error;

        public TaskResultFutureAdapter(T result, Throwable error) {
            this._result = result;
            this._error = error;
        }

        @Override
        public boolean cancel(boolean mayInterruptIfRunning) {
            return false;
        }

        @Override
        public boolean isCancelled() {
            return false;
        }

        @Override
        public boolean isDone() {
            return true;
        }

        @Override
        public T get() throws ExecutionException {
            if (this._error != null) {
                throw new ExecutionException(this._error);
            }
            return this._result;
        }

        @Override
        public T get(long timeout, TimeUnit unit) throws ExecutionException {
            return this.get();
        }
    }

    public static enum ErrorHandlingPolicy {
        STRICT,
        LENIENT;

    }
}

