/*
 * Decompiled with CFR 0.152.
 */
package com.vmware.ph.client.api.impl.aggregation;

import com.vmware.ph.client.api.impl.aggregation.Buffer;
import com.vmware.ph.common.compress.GzipUtil;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import org.apache.http.annotation.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class RotatingFileBuffer
implements Buffer,
Closeable {
    private static final long DEFAULT_CACHE_FILE_SIZE_BYTES = 0x500000L;
    private static final int DEFAULT_NUMBER_OF_CACHE_FILES = 5;
    static final String DEFAULT_CACHE_FILE_NAME_PREFIX = "cachefile";
    static final String DEFAULT_BUFFER_IDENTIFIER = "aggregation";
    private static final Logger log = LoggerFactory.getLogger(RotatingFileBuffer.class);
    private final long cacheFileSizeBytes;
    private final int maxNumberOfCacheFiles;
    private final String cacheFileNamePrefix;
    private final File[] cacheFiles;
    private final File tempFile;
    private OutputStream currentFileOutputStream;
    private InputStream lastFileInputStream;
    private boolean isCurrentlyReading = false;
    private long currentFileSize;
    private int currentCacheFileIndex;
    private final FileTransformer fileTransformer;
    private final SneakernetCaseImplementation sneakernet;

    public RotatingFileBuffer(File cacheFolder, HeaderProvider hp) {
        this(cacheFolder, hp, 0x500000L, 5);
    }

    public RotatingFileBuffer(File cacheFolder, HeaderProvider hp, long cacheFileSizeBytes, int numberOfCacheFiles) {
        this(cacheFolder, DEFAULT_BUFFER_IDENTIFIER, cacheFileSizeBytes, numberOfCacheFiles, DEFAULT_CACHE_FILE_NAME_PREFIX, new GzipFileTransformer(hp), new SneakernetCaseImplementation(hp, cacheFolder));
    }

    RotatingFileBuffer(File cacheFolder, String bufferIdentifier, long cacheFileSizeBytes, int numberOfCacheFiles, String cacheFileNamePrefix, FileTransformer fileTransformer, SneakernetCaseImplementation sneakernet) {
        this.cacheFileSizeBytes = cacheFileSizeBytes;
        this.maxNumberOfCacheFiles = numberOfCacheFiles;
        this.cacheFileNamePrefix = cacheFileNamePrefix;
        this.fileTransformer = fileTransformer;
        this.sneakernet = sneakernet;
        cacheFolder.mkdirs();
        this.cacheFiles = new File[numberOfCacheFiles];
        for (int i = 0; i < numberOfCacheFiles; ++i) {
            this.cacheFiles[i] = new File(cacheFolder, this.cacheFileNamePrefix + '_' + bufferIdentifier + '_' + i);
        }
        this.tempFile = new File(cacheFolder, bufferIdentifier + "_tmp");
        this.currentFileSize = 0L;
        this.initialize();
    }

    @Override
    public synchronized void writeData(byte[] b, int off, int len) throws IOException {
        log.trace("Writing a data chunk with size {} ...", (Object)len);
        if (this.currentFileOutputStream == null) {
            try {
                this.currentFileOutputStream = new FileOutputStream(this.cacheFiles[this.currentCacheFileIndex], true);
            }
            catch (FileNotFoundException e) {
                log.error("While writing data to the buffer, FileOutputStream couldn't be open. As a result the buffer can't continue working normally and will exit. The data chunk will be lost, however the existing buffer data will be used next time the buffer is created. Check the exception for the root cause, probably there is a problem with creating the file such as insufficient disk space, write access violation, etc.", (Throwable)e);
                throw new RuntimeException("Can't open cache file for writing.", e);
            }
        }
        this.currentFileOutputStream.write(b, off, len);
        this.currentFileOutputStream.flush();
        this.currentFileSize += (long)len;
        if (null != this.sneakernet) {
            this.sneakernet.createHeaderFileIfNeeded();
        }
        if (this.currentFileSize >= this.cacheFileSizeBytes) {
            log.debug("The size limit of the file {} has been exceeded, new file will be created.", (Object)this.cacheFiles[this.currentCacheFileIndex]);
            this.splitCurrentFile();
        }
    }

    @Override
    public synchronized boolean isReadyToConsume() {
        return this.currentCacheFileIndex > 0;
    }

    @Override
    public synchronized Buffer.BufferedData getBufferedData() {
        log.debug("Getting data for reading...");
        if (this.isCurrentlyReading) {
            log.error("A repeated attempt was made to obtain data for reading from the buffer but data has not been released from its first attempt. Returning null. Data must be released after reading is finished");
            return null;
        }
        if (this.currentCacheFileIndex == 0) {
            if (this.cacheFiles[0].exists()) {
                log.debug("There's only one cache file. A new file will be created so that the first one is consumed.");
                this.splitCurrentFile();
            } else {
                log.info("No data has been written yet, returning null.");
                return null;
            }
        }
        try {
            long dataLength;
            if (this.fileTransformer != null) {
                if (!this.tempFile.exists()) {
                    log.debug("Applying data transformation to the file...");
                    this.fileTransformer.transformFile(this.cacheFiles[0], this.tempFile);
                    log.debug("File successfully tranformed.");
                }
                this.lastFileInputStream = new FileInputStream(this.tempFile);
                dataLength = this.tempFile.length();
            } else {
                this.lastFileInputStream = new FileInputStream(this.cacheFiles[0]);
                dataLength = this.cacheFiles[0].length();
            }
            this.isCurrentlyReading = true;
            log.debug("File input stream has been open, it will be returned.");
            return new Buffer.BufferedData(this.lastFileInputStream, dataLength);
        }
        catch (IOException e) {
            log.error("While preparing data to consume there was an IO exception. Returning null. See the root cause exception, probably couldn't tranform the file or open it for reading.", (Throwable)e);
            return null;
        }
    }

    @Override
    public synchronized void releaseData(boolean isSuccessfullyRead) {
        log.debug("Releasing data, reading was successful: {}", (Object)isSuccessfullyRead);
        this.isCurrentlyReading = false;
        this.closeLastFileInputStream();
        if (isSuccessfullyRead) {
            log.debug("Reading was successful, so the top file will be removed.");
            this.rotateFiles();
            this.deleteTempFile();
            if (null != this.sneakernet && 0 == this.currentCacheFileIndex) {
                if (!this.cacheFiles[0].exists()) {
                    this.sneakernet.deleteHeaderFile();
                } else {
                    this.sneakernet.createHeaderFile();
                }
            }
        }
    }

    private synchronized void splitCurrentFile() {
        if (this.currentCacheFileIndex == this.maxNumberOfCacheFiles - 1) {
            log.debug("MAX number of cache files {} has been reached, the oldest one will be removed", (Object)this.maxNumberOfCacheFiles);
            this.rotateFiles();
        }
        if (this.cacheFiles[this.currentCacheFileIndex].exists()) {
            this.closeCurrentFileOutputStream();
            ++this.currentCacheFileIndex;
            this.currentFileSize = 0L;
        }
    }

    private synchronized void rotateFiles() {
        log.debug("Removing oldest cache file, current cache file index: {}, current cache file exists: {}", (Object)this.currentCacheFileIndex, (Object)this.cacheFiles[this.currentCacheFileIndex].exists());
        int fromIndex = this.isCurrentlyReading ? 1 : 0;
        log.debug("The file to be removed has index {}", (Object)fromIndex);
        this.closeCurrentFileOutputStream();
        for (int i = fromIndex; i < this.currentCacheFileIndex; ++i) {
            if (this.cacheFiles[i + 1].exists()) {
                Path from = this.cacheFiles[i + 1].toPath();
                Path to = this.cacheFiles[i].toPath();
                log.debug("Renaming file {} to {}...", (Object)from, (Object)to);
                try {
                    Files.move(from, to, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
                    log.debug("Rename successful.");
                    continue;
                }
                catch (IOException e) {
                    log.error("While rotating cache files, couldn't rename file " + from + " to " + to + ". The buffer can't continue working correctly and will exit. " + "On next start it will continue where it left. Check the root cause exception, " + "there may be insufficient disk space ot write access violation.", (Throwable)e);
                    throw new RuntimeException("Can't rotate cache files.", e);
                }
            }
            if (!this.cacheFiles[i].exists()) continue;
            log.debug("Deleting cache file {}", (Object)this.cacheFiles[i]);
            try {
                Files.delete(this.cacheFiles[i].toPath());
                continue;
            }
            catch (IOException e) {
                throw new RuntimeException("Can't delete file", e);
            }
        }
        --this.currentCacheFileIndex;
    }

    private synchronized void closeCurrentFileOutputStream() {
        if (this.currentFileOutputStream == null) {
            return;
        }
        try {
            this.currentFileOutputStream.close();
            this.currentFileOutputStream = null;
        }
        catch (IOException e) {
            log.warn("Can't close current output stream", (Throwable)e);
        }
    }

    private synchronized void closeLastFileInputStream() {
        if (this.lastFileInputStream != null) {
            try {
                this.lastFileInputStream.close();
                this.lastFileInputStream = null;
            }
            catch (IOException e) {
                log.warn("Can't close latest FileInputStream.", (Throwable)e);
            }
        }
    }

    private synchronized void deleteTempFile() {
        if (this.fileTransformer != null && this.tempFile.exists()) {
            try {
                log.debug("Deleting temp file {}", (Object)this.tempFile);
                Files.delete(this.tempFile.toPath());
            }
            catch (IOException e) {
                log.warn("While deleting temp file that is used for transformation, there was IO error. The file will remain on the filesystem, however a new attempt to delete it will be performed next time files are rotated.", (Throwable)e);
            }
        }
    }

    private void initialize() {
        log.debug("Initializing the buffer and checking for exisitng files...");
        this.currentFileSize = 0L;
        this.currentCacheFileIndex = -1;
        for (File cacheFile : this.cacheFiles) {
            if (!cacheFile.exists()) continue;
            ++this.currentCacheFileIndex;
            this.currentFileSize = cacheFile.length();
            log.debug("Found existing cache file {}, size {}, current file index {}", new Object[]{cacheFile, this.currentFileSize, this.currentCacheFileIndex});
        }
        if (this.currentCacheFileIndex == -1) {
            this.currentCacheFileIndex = 0;
            log.debug("No existing cache files found");
        }
        if (this.currentCacheFileIndex < this.maxNumberOfCacheFiles - 1 && this.currentFileSize >= this.cacheFileSizeBytes) {
            log.debug("The latest existing cache file found is already full. The next write operation will create a new file and will write in it.");
            ++this.currentCacheFileIndex;
            this.currentFileSize = 0L;
        }
    }

    @Override
    public void close() throws IOException {
        this.closeCurrentFileOutputStream();
        this.closeLastFileInputStream();
        this.deleteTempFile();
        if (null != this.sneakernet) {
            this.sneakernet.deleteHeaderFile();
        }
    }

    public static interface HeaderProvider {
        public byte[] getHeader();

        public static class FixedOne
        implements HeaderProvider {
            private final byte[] header;

            public FixedOne(byte[] header) {
                this.header = header;
            }

            @Override
            public byte[] getHeader() {
                return this.header;
            }
        }
    }

    static class SneakernetCaseImplementation
    implements AutoCloseable {
        private final HeaderProvider headerProvider;
        private final File folder;
        private final String absolutePath;
        private Boolean headerFileAlreadyExists;
        private final Object headerFileAlreadyExistsLock = new Object();

        SneakernetCaseImplementation(HeaderProvider hp, File folder) {
            assert (null != hp);
            this.headerProvider = hp;
            this.folder = folder;
            this.absolutePath = folder.getAbsolutePath().intern();
            this.headerFileAlreadyExists = false;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void createHeaderFileIfNeeded() {
            Object object = this.headerFileAlreadyExistsLock;
            synchronized (object) {
                if (!this.headerFileAlreadyExists.booleanValue()) {
                    this.createHeaderFile();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void createHeaderFile() {
            assert (this.folder.isDirectory());
            File f = this.getHeaderFile();
            String string = this.absolutePath;
            synchronized (string) {
                try (FileOutputStream out = new FileOutputStream(f);){
                    out.write(this.headerProvider.getHeader());
                    Object object = this.headerFileAlreadyExistsLock;
                    synchronized (object) {
                        this.headerFileAlreadyExists = true;
                    }
                }
                catch (IOException e) {
                    String msg = "Writing header file to buffer folder " + this.folder + ". This means that sneakernet case may not work." + " Everything else remains unaffected." + " Program continues.";
                    log.error(msg, (Throwable)e);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void deleteHeaderFile() {
            String string = this.absolutePath;
            synchronized (string) {
                this.getHeaderFile().delete();
                Object object = this.headerFileAlreadyExistsLock;
                synchronized (object) {
                    this.headerFileAlreadyExists = false;
                }
            }
        }

        @Override
        public void close() throws Exception {
            this.deleteHeaderFile();
        }

        File getHeaderFile() {
            return new File(this.folder, "header");
        }
    }

    static class GzipFileTransformer
    implements FileTransformer {
        private final HeaderProvider fileHeaderProvider;

        GzipFileTransformer(HeaderProvider fileHeaderProvider) {
            this.fileHeaderProvider = fileHeaderProvider;
        }

        @Override
        public void transformFile(File input, File output) throws IOException {
            byte[] buffer = new byte[1024];
            try (FileInputStream in = new FileInputStream(input);
                 FileOutputStream out = new FileOutputStream(output);){
                int len;
                if (this.fileHeaderProvider != null) {
                    byte[] res = this.fileHeaderProvider.getHeader();
                    ((OutputStream)out).write(res);
                }
                while ((len = ((InputStream)in).read(buffer)) != -1) {
                    ((OutputStream)out).write(buffer, 0, len);
                }
            }
            File temp = new File(output.getParentFile(), output.getName() + ".gzip");
            GzipUtil.gzipFile(output, temp);
            Files.move(temp.toPath(), output.toPath(), StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        }
    }

    static interface FileTransformer {
        public void transformFile(File var1, File var2) throws IOException;
    }
}

