/***********************************************************
 * Copyright 2011 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.sso;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vide.sso.model.Token;

public class TokenStorageManager {
    private static final Logger logger = LoggerFactory.getLogger(TokenStorageManager.class);
    private static final int SIZE_THRESHOLD = 32;

    private Map<String, Token> tokens;

    private static class AuthStorageManagerHolder {
        private static final TokenStorageManager INSTANCE = new TokenStorageManager();
    }

    public static TokenStorageManager getInstance() {
        return AuthStorageManagerHolder.INSTANCE;
    }

    private TokenStorageManager() {
        tokens = new HashMap<String, Token>();
    }

    public boolean isExpired(Token token) {
        return System.currentTimeMillis() > token.getExpiration();
    }

    public synchronized Token get(String key) {
        return tokens.get(key);
    }

    private void cleanExpiredTokens() {
        Iterator<Map.Entry<String, Token>> iterator = tokens.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, Token> entry = iterator.next();
            if (isExpired(entry.getValue())) {
                logger.info("Removing expired token {}", entry.getValue().getAuthToken());
                iterator.remove();
            }
        }
    }

    public synchronized Token put(String key, Token value) {
        Token token = tokens.put(key, value);
        if (tokens.size() > SIZE_THRESHOLD) {
            cleanExpiredTokens();
        }
        return token;
    }
}
