/***********************************************************
 * Copyright 2011 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.security.CodeSource;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vide.vlogbrowser.core.consts.LogBrowserConsts;
import com.vmware.vide.vlogbrowser.core.fileops.LogFileManager;
import com.vmware.vide.vlogbrowser.core.model.LogSystemType;
import com.vmware.vide.vlogbrowser.core.model.LogxFile;
import com.vmware.vide.vlogbrowser.core.parser.ConfigParser;
import com.vmware.vide.vlogbrowser.core.parser.LogFormat;
import com.vmware.vide.vlogbrowser.core.parser.RAFile;
import com.vmware.vide.vlogbrowser.core.utils.ArchiveUtils;

public abstract class AbstractLogBrowserManager implements ILogBrowserManager {
    // Stores LogxFile objects, representing the .logx files found in .../templates/
    protected Vector<LogxFile> logxTemplates;

    // A list of the Files that have been scanned in both system and user-provided locations
    protected List<File> logxTemplateFiles;

    // Each time a host is probed, its type is stored here to avoid costly re-probing operations
    protected Map<String, String> hostType;

    protected Map<String, LogxFile[]> allFormats;   // All Log formats for a given host (the key)
    protected Map<String, LogxFile[]> absolutePathFormats;  // only log formats with a static path

    /** Directories with custom user template files */
    protected String[] fCustomTemplateDirs;

    protected String fRecentHost;
    protected Map<String, String> fRecentHostDirs;
    protected Map<String, String> fRecentHostFormats;
    
    private static final Logger logger = LoggerFactory.getLogger(AbstractLogBrowserManager.class);

    public AbstractLogBrowserManager() {
        logxTemplates = new Vector<LogxFile>();
        hostType = new HashMap<String, String>();

        this.allFormats = new HashMap<String, LogxFile[]>();
        this.absolutePathFormats = new HashMap<String, LogxFile[]>();
        fCustomTemplateDirs = new String[0];
        fRecentHostDirs = new HashMap<String, String>();
        fRecentHostFormats = new HashMap<String, String>();
    }

    @Override
    public String[] getCustomTemplateDirs() {
        return fCustomTemplateDirs;
    }

    @Override
    public void setCustomTemplateDirs(String[] dirs) {
        fCustomTemplateDirs = dirs;

        // force re-scanning templates on next use
        logxTemplateFiles = null;
        logxTemplates.clear();
        absolutePathFormats.clear();
        allFormats.clear();
    }

    @Override
    public void setRecentHostDir(String hostname, String lastDirectory) {
        fRecentHostDirs.put(hostname, lastDirectory);
    }

    @Override
    public String getRecentHostDir(String hostname) {
        String value = fRecentHostDirs.get(hostname);
        if (value == null)
            value = "";
        return value;
    }

    @Override
    public void setRecentHostFormat(String hostname, String lastDirectory) {
        fRecentHostFormats.put(hostname, lastDirectory);
    }

    @Override
    public String getRecentHostFormat(String hostname) {
        String value = fRecentHostFormats.get(hostname);
        if (value == null)
            value = "";
        return value;
    }

    @Override
    public String getRecentHost() {
        return fRecentHost;
    }

    @Override
    public void setRecentHost(String recentHost) {
        fRecentHost = recentHost;
    }

    protected File[] getFilesInDirectory(String path, final String extension) throws Exception {
        if (path != null) {
            File file = new File(path);
            if (file.exists() && file.isDirectory()) {
                return file.listFiles(new FileFilter() {
                    @Override
                    public boolean accept(File path) {
                        return path.isFile() && path.getName().endsWith(extension);
                    }
                });
            }
        }
        return new File[0];
    }

    protected Collection<URL> getCompleteTemplateFileList() throws IOException {
        Collection<URL> result = new ArrayList<URL>();
        CodeSource src = this.getClass().getProtectionDomain().getCodeSource();
        if( src != null ) {
            URL url = src.getLocation();
            File file = new File(url.getFile());
            boolean isJar = file.isFile() && file.getAbsolutePath().endsWith(".jar");
            if (isJar) {
                ZipInputStream zip = new ZipInputStream( url.openStream());
                ZipEntry ze = null;

                while( ( ze = zip.getNextEntry() ) != null ) {
                    String entryName = ze.getName();
                    if(entryName.endsWith(LogBrowserConsts.LOGX_FILE_EXTENSION)) {
                        result.add(getClass().getResource("/"+entryName));
                    }
                }
            } else {
                File fPath = new File(url.getFile(), LogBrowserConsts.TEMPLATE_DIR);
                if (fPath.exists() && fPath.isDirectory()) {
                    File[] files = fPath.listFiles(new FileFilter() {
                        @Override
                        public boolean accept(File file) {
                            return file.isFile() && file.getName().endsWith(LogBrowserConsts.LOGX_FILE_EXTENSION);
                        }
                    });
                    for (File f: files) {
                        result.add(f.toURI().toURL());
                    }
                }
            }
            
           
            
            
            
         }
        return result;
    }

    protected void parseLogxTemplates() throws Exception {
        Collection<URL> logFormatFiles = getCompleteTemplateFileList();
        for (URL url : logFormatFiles) {
            InputStream fstrm = url.openStream();
            ConfigParser cfgParser = new ConfigParser(fstrm);
            String logFormatName = cfgParser.getLogFormatName();
            String category = cfgParser.getCategory();
            LogSystemType systemType = cfgParser.getSystemType();
            LogxFile template = new LogxFile(logFormatName, category, url, systemType);
            logxTemplates.add(template);
            fstrm.close();
        }
    }

    @Override
    public boolean isShowAllTemplates(String hostname) {
//        ITargetManager targetMgr = TargetManagerPlugin.getTheTargetManager();
//        ITarget target = targetMgr.findTarget(hostname);
//        if (target == null)
//            return false;
//
//        String targetType = target.getTargetType();
//        String label = target.getLabel();
//        return targetType.equals(ITargetManager.TARGET_TYPE_LOCAL)
//                || (label != null && label.equals("SSH Only"));
        return false;
    }

    @Override
    public LogxFile[] getLogFormatsByHosttype(String hostType, String hostVer, boolean knownPathOnly) throws Exception {
        LogxFile[] result;
        ArrayList<LogxFile> matchedTemplates = new ArrayList<LogxFile>();

        // Read and parse the .logx templates on first use
        if (logxTemplates.isEmpty())
            parseLogxTemplates();

        // scan templates for system type match
        for (int i = 0; i < logxTemplates.size(); i++) {
            // check if template system type matches host
            LogxFile template = logxTemplates.get(i);
            String system = template.getSystemType().getSystem();
            String version = template.getSystemType().getVersion();

            // check if system type matches or empty
            boolean sysMatches = system.length() == 0 || Pattern.matches(system.toLowerCase(), hostType.toLowerCase());
            boolean verMatches = version.length() == 0 || hostVer.length() == 0 ||
                Pattern.matches(version.toLowerCase(), hostVer.toLowerCase());

            // enforce system and version match for ESX hosts,
            // for other hosts (like Linux SSH) only enforce knownPathOnly at this time
            if (!sysMatches || !verMatches)
                continue;

            // check for absolute path templates
            String defParDir = template.getSystemType().getDir();
            boolean isAbsolutePath = defParDir.startsWith("/") || defParDir.startsWith(":\\", 1);
            if (isAbsolutePath && knownPathOnly || !knownPathOnly)
                matchedTemplates.add(template);
        }

        result = matchedTemplates.toArray(new LogxFile[0]);
        return result;
    }

    @Override
    public LogxFile[] getLogFormatsByHosttype(LogSystemType tgzSysType,
            String tgzFilepath) throws Exception {
        LogxFile[] result;
        ArrayList<LogxFile> matchedTemplates = new ArrayList<LogxFile>();

        // Read and parse the .logx templates on first use
        if (logxTemplates.isEmpty())
            parseLogxTemplates();

        String hostType = tgzSysType.getSystem();
        String hostVer = tgzSysType.getVersion();

        // scan templates for system type match
        for (int i = 0; i < logxTemplates.size(); i++) {
            // check if template system type matches host
            LogxFile template = logxTemplates.get(i);
            String system = template.getSystemType().getSystem();
            String version = template.getSystemType().getVersion();

            // check if system type matches or empty
            boolean sysMatches = system.length() == 0
                    || Pattern.matches(system.toLowerCase(),
                            hostType.toLowerCase());
            boolean verMatches = version.length() == 0
                    || hostVer.length() == 0
                    || Pattern.matches(version.toLowerCase(),
                            hostVer.toLowerCase());

            // enforce system and version match for ESX hosts,
            // for other hosts (like Linux SSH) only enforce knownPathOnly at
            // this time
            if (!sysMatches || !verMatches)
                continue;

            Collection<Pattern> patterns = new ArrayList<Pattern>();

            String[] paths = template.getSystemType().getDir().trim().split(Pattern.quote("|"));
            for (String path : paths) {
                String filename = template.getSystemType().getLiveLogFilename().trim();
                Pattern pattern = Pattern.compile(".*" + Pattern.quote(path.trim()) + filename);
                patterns.add(pattern);
            }

            String match = ArchiveUtils.searchFileNamePatterns(new File(
                    tgzFilepath), patterns, true);

            if (match != null) {
                matchedTemplates.add(template);
            }
        }

        result = matchedTemplates.toArray(new LogxFile[0]);
        return result;
    }

    @Override
    public List<LogFormat> getLogFormats() throws Exception {
        List<LogFormat> logFormats = new ArrayList<LogFormat>(10);
        Collection<URL> logFormatFiles = getCompleteTemplateFileList();
        for (URL url : logFormatFiles) {
            InputStream fstrm = url.openStream();
            ConfigParser cfgParser = new ConfigParser(fstrm);
                logFormats.add(cfgParser.getLogFormat());
            fstrm.close();
        }
        return logFormats;
    }

    @Override
    public URL getLogxTemplateFileURL(String name, String version) throws Exception {
        // find first template with specified name and supported version
        if (logxTemplates.isEmpty())
            parseLogxTemplates();

        for (int i = 0; i < logxTemplates.size(); i++) {
            LogxFile template = logxTemplates.get(i);

            // check name
            String templateName = template.getLogFormatName();
            boolean nameMatch = (name != null && name.compareToIgnoreCase(templateName) == 0);

            // check version
            String templateVersion = template.getSystemType().getVersion();
            boolean versionMatch = version.length() == 0 || templateVersion.length() == 0 ||
                Pattern.matches(templateVersion.toLowerCase(), version.toLowerCase());

            if (nameMatch && versionMatch) {
//                String path = template.getTemplateFilePath();
//                File f = new File(path);
//                return f;
                return template.getTemplateURL();
            }
        }

        return null;
    }

    @Override
    public void clearHostFormats(String host) {
        allFormats.remove(host);
        absolutePathFormats.remove(host);
    }

    private String getValueFromRegexGroup(String regex, int groupIndex, String s) {
        Pattern p = Pattern.compile(regex);
        Matcher m = p.matcher(s);
        if (m.find()) {
            return m.group(groupIndex);
        }
        return "";
    }

    private String getHostnamefromFilePath(String path) {
        // reconstruct host name from path, use regular expression to parse
        // hostname, path examples:
        // vm-support-hv-peng161-2010-09-14--14.15.6999/etc/vmware/esx.conf
        // - ESX 4
        // vm-support-cde-dev1-2010-12-22--16.19.90425181 - ESXi 4
        // esx-peng013.eng.vmware.com-2010-11-11--16.45/etc/vmware/esx.conf
        // - ESXi 5
        String hostname = null;
        String regex = "(vm-support-|esx-|vc-)(.+)\\-\\d\\d\\d\\d-\\d\\d-\\d\\d--.*";
        Pattern p = Pattern.compile(regex);
        Matcher m = p.matcher(path);
        if (m.find()) {
            // get hostname from match result
            hostname = m.group(2);

            // remove domain name
            int dotIndex = hostname.indexOf(".");
            if (dotIndex > 0) {
                hostname = hostname.substring(0, dotIndex);
            }
        }
        return hostname;
    }

    @Override
    public LogSystemType getArchiveFileType(String host, String archiveFilePath)
            throws Exception {
        logger.debug("Scanning \".tgz\" file to determine its type...", 3);
        boolean hasVmkernel1 = false; // at /var/log
        boolean hasVmkernel2 = false; // at /var/run/log
        boolean hasFdm1 = false; // at /var/log
        boolean hasFdm2 = false; // at /var/run/log
        String hostname = host;

        logger.debug("Checking VC...");
        if (LogFileManager.isLocalHost(host)) {
            String VC_REGEX = ".*VMware\\sVirtualCenter\\sServer[/\\\\](\\d).*";
            File archiveFile = new File(archiveFilePath);

            String match = ArchiveUtils.searchFileNamePatterns(archiveFile,
                    Collections.singleton(Pattern.compile(VC_REGEX)), false);
            // It is a VC
            if (match != null) {
                String version = getValueFromRegexGroup(VC_REGEX, 1, match);
                String hostFromPath = getHostnamefromFilePath(match);
                if (hostFromPath != null) {
                    hostname = hostFromPath;
                }

                if (getValueFromRegexGroup("(ProgramData)", 1, match).isEmpty()) {
                    // Linux
                    return new LogSystemType(LogBrowserConsts.VC_LINUX, version+"*", hostname,
                          "", "", false);
                } else {
                    // Windows
                    return new LogSystemType(LogBrowserConsts.VC_WINDOWS, version+"*", hostname,
                            "", "", false);
                }
            }
        }

        // check for /var/log/vmkernel
        logger.debug("Checking vmkernel log at /var/log ...");
        List<RAFile> raFiles = LogFileManager.getMatchingFiles(host,
                archiveFilePath, "/var/log/", "vmkernel", true);
        hasVmkernel1 = raFiles.size() > 0;

        if (!hasVmkernel1) {
            logger.debug("Checking vmkernel log at /var/run/log ...");
            raFiles = LogFileManager.getMatchingFiles(host,
                    archiveFilePath, "/var/run/log/", "vmkernel", true);
            hasVmkernel2 = raFiles.size() > 0;
        }

        // check for /var/log/fdm.log - ESXi-5 only
        logger.debug("Checking fdm log at /var/log ...");
        raFiles = LogFileManager.getMatchingFiles(host, archiveFilePath,
                "/var/log/", "fdm", true);
        hasFdm1 = raFiles.size() > 0;

        if (!hasFdm1) {
            logger.debug("Checking fdm log at /var/run/log ...");
            raFiles = LogFileManager.getMatchingFiles(host, archiveFilePath,
                    "/var/run/log/", "fdm", true);
            hasFdm2 = raFiles.size() > 0;
        }

        // check for esx.conf, this file contains hostname,
        // but hostname can be found from file path too w/o reading file
        logger.debug("Checking esx.conf...");
        raFiles = LogFileManager.getMatchingFiles(host, archiveFilePath,
                "/etc/vmware/", "esx.conf", true);

        if (raFiles.size() > 0) {
            RAFile confFile = raFiles.get(0);
            hostname = getHostnamefromFilePath(confFile.getSrcFilePath());
        }

        String system = null;
        String version = "";

        // check if ESX-4
        if (hasVmkernel1 && !hasFdm1) {
            system = LogBrowserConsts.ESX;
            version = "4*";
        }

        // check if ESXi-4
        if (!hasVmkernel1 && !hasFdm1 && !hasVmkernel2 && !hasFdm2) {
            system = LogBrowserConsts.ESXi;
            version = "4*";
        }

        // check if ESXi-5
        if (hasVmkernel1 && hasFdm1) {
            system = LogBrowserConsts.ESXi;
            version = "5*";
        }

        // check if ESXi-6
        if (!hasVmkernel1 && !hasFdm1 && hasVmkernel2 && hasFdm2) {
            system = LogBrowserConsts.ESXi;
            version = "6*";
        }

        return new LogSystemType(system, version, hostname, "", "", false);
    }
}
