/***********************************************************
 * Copyright 2011 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core;

import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.List;

import com.vmware.vide.vlogbrowser.core.model.LogSystemType;
import com.vmware.vide.vlogbrowser.core.model.LogxFile;
import com.vmware.vide.vlogbrowser.core.parser.LogFormat;

public interface ILogBrowserManager {
    /**
     * Returns list of folders with user-provided log templates
     * @return
     */
    public String[] getCustomTemplateDirs();

    /**
     * Returns list of folders with user-provided log templates
     * @param dirs
     */
    public void setCustomTemplateDirs(String[] dirs);

    /**
     * Save preferences
     */
    public void savePreferences();

    /**
     * Load preferences
     */
    public void loadPreferences();

    /**
     * Set recent host directory
     * @param hostname
     * @param lastDirectory
     */
    public void setRecentHostDir(String hostname, String lastDirectory);

    /**
     * Get recent host directory
     * @param hostname
     * @return
     */
    public String getRecentHostDir(String hostname);

    /**
     * Set recent host format
     * @param hostname
     * @param lastDirectory
     */
    public void setRecentHostFormat(String hostname, String lastDirectory);

    /**
     * Get recent host format
     * @param hostname
     * @return
     */
    public String getRecentHostFormat(String hostname);

    /**
     * Get recent host
     * @return
     */
    public String getRecentHost();

    /**
     * Set recent host
     * @param recentHost
     */
    public void setRecentHost(String recentHost);

    /**
     * Given an RSE connection name, return the list of log formats supported by that host. The
     * first time a query comes in for a particular connection name, that information is
     * retrieved based on the type of host and stored in one of TWO hashmaps for caching
     * purposes; one hashmap stores a mapping of each hostname to all compatible log formats,
     * and the other stores a mapping of each hostname to the log formats that have absolute
     * paths (for formats with absolute paths, we can default the parent directory without
     * requiring input from the user). Select between these two hashmaps using the
     * "knownPathOnly" variable.
     * @param connectionName
     * @param knownPathOnly
     * @return
     */
    public LogxFile[] getLogFormatsByHostService(String connectionName, boolean knownPathOnly);

    /**
     * Given an RSE connection name, return the list of log formats supported by that host. The
     * first time a query comes in for a particular connection name, that information is
     * retrieved based on the type of host and stored in one of TWO hashmaps for caching
     * purposes; one hashmap stores a mapping of each hostname to all compatible log formats,
     * and the other stores a mapping of each hostname to the log formats that have absolute
     * paths (for formats with absolute paths, we can default the parent directory without
     * requiring input from the user). Select between these two hashmaps using the
     * "knownPathOnly" variable.
     * @param connectionName
     * @param knownPathOnly
     * @return
     */
    public LogxFile[] getLogFormatsByHost(String connectionName, boolean knownPathOnly);

    /**
     * Returns whether all templates should be shown for the selected host.
     * @param hostname
     * @return true for Local or SSH Only, false otherwise
     */
    public boolean isShowAllTemplates(String hostname);

    /**
     * Returns list of log formats supported by host type and version. This is the only place
     * where hosttype of log template is compared with type of the host for which templates are
     * requested.
     * @param hostType host type as reported by RSE
     * @param hostVer host version (ESX version)
     * @param knownPathOnly only add logs with known path
     * @return array of log templates for this host type
     * @throws Exception exceptions from RSE if any
     */
    public LogxFile[] getLogFormatsByHosttype(String hostType, String hostVer,
            boolean knownPathOnly) throws Exception;

    /**
     * Returns list of log formats supported by host type and version. This is the only place
     * where hosttype of log template is compared with type of the host for which templates are
     * requested.
     * @param tgzSysType the host's system type
     * @param tgzFilepath host's bundle file path
     * @return array of log templates for this host type
     * @throws Exception exceptions from RSE if any
     */
    public LogxFile[] getLogFormatsByHosttype(LogSystemType tgzSysType, String tgzFilepath) throws Exception;

    /**
     * Returns all log format templates
     * @return
     * @throws Exception
     */
    public List<LogFormat> getLogFormats() throws Exception;

    /**
     * Finds and returns template file with specified name and ESX version. Templates with same
     * name can be different for different host typer and versions (e.g. hostd and vmkernel are
     * different between ESX 4 and 5), therefore version string to be provided
     * @param name case-insensitive template name to find
     * @param version host version to match, if null then only name is matched
     * @return found template file or null
     */
    public URL getLogxTemplateFileURL(String name, String version) throws Exception;

    /**
     * The Activator object stores a HashMap of <String, String>, i.e. ConnectionName, HostType.
     * This method is used to retrieve the host type string for a given connection name. For
     * better efficiency, the RSE is not queried more than necessary with expensive operations
     * like "isVisor()". Instead, the host type is cached here in the Activator object's
     * "hostType" HashMap object.
     * @param target The RSE target to query
     * @param retryOnError A remote connection should be attempted even though the target was
     *            down during the last attempt.
     * @return The host type as a String
     */
//    public String getHostType(final ITarget target, boolean retryOnError);

    /**
     * Removes cached log formats for the specified host, to force update
     * @param host
     */
    public void clearHostFormats(String host);

    /**
     * The user can choose a vm-support log archive (a .tgz file) located on the local system,
     * or on a remote system. LogSystemType returned by this method contains following info:
     * LogSystemType.system - system type LogSystemType.version - system version
     * LogSystemType.parentDir - hostname where vm-support file was generated on
     * @param host
     * @param archiveFilePath
     * @return
     */
    public LogSystemType getTgzFileType(final String host, final String archiveFilePath);

    /**
     * Inspect the content of the archive file to detect the system type (ESX,ESXi,VC)
     * and version of the system that generated the archive.
     * @param host Name of host in case of remote system
     * @param archiveFilePath Path to vmsupport bundle
     * @param monitor
     * @return
     * @throws InvocationTargetException
     * @throws InterruptedException
     */
    LogSystemType getArchiveFileType(String host, String archiveFilePath) throws Exception;
}
