/***********************************************************
 * Copyright 2009-2011 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core.fileops;

import java.util.List;
import java.util.concurrent.Future;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vide.vlogbrowser.core.model.FilterList;
import com.vmware.vide.vlogbrowser.core.model.ILogItemList;
import com.vmware.vide.vlogbrowser.core.model.LogItem;
import com.vmware.vide.vlogbrowser.core.parser.LogFormat;
import com.vmware.vide.vlogbrowser.core.parser.LogParser;
import com.vmware.vide.vlogbrowser.core.parser.RAFile;
import com.vmware.vide.vlogbrowser.core.utils.TaskScheduler;

/**
 * The LogWorker class is called by the LogTableViewer object and used to spawn a different
 * job (thread) to do the work of loading a log file from disk and parsing it using a LogParser
 * object. That LogParser object reads a List of LogItems, then calls the loadLogItems() method
 * in the LogWorker to load those LogItems into the JFace TableViewer.
 */
public class BaseLogWorker {

    protected FilterList filterList;
    protected LogFormat logFormat;
    protected String logWorkerType;
//    private ReentrantLock refreshLock;
    private LogFileManager logFileMgr;

    protected final String FILE_READ_ERROR = "File read error";
    protected final String FILE_ERROR_MSG = "The file could not be read.";
    protected final String FILE_COPY_ERROR = "File copy error";
    protected final String COPY_ERROR_MSG =
        "The file could not be copied from the source to the staging area.";

    private boolean working = false;
    
    private static final Logger logger = LoggerFactory.getLogger(BaseLogWorker.class);

    public BaseLogWorker() {
    }

    /**
     * Allow the other job to discover about this cancellation, so they can cancel
     * itself as well.
     * Override in derived classes based on UI.
     */
//    public boolean areAllJobsCanceled() {
//        return false;
//    }

    public void marshalLogFiles(final List<RAFile> raFiles) {
        if (!logWorkerType.equals("FILE")) {
            return;
        }
//        Job job = new Job("Marshalling " + logFormat.getName() + " log files, please wait") {
//
//            protected IStatus run(IProgressMonitor monitor) {
//                IStatus returnStatus = Status.OK_STATUS;
//                try {
//                    monitor.beginTask("Copying log files to staging directory", raFiles.size() + 1);
//                    logFileMgr.stageLogFiles(raFiles, monitor, BaseLogWorker.this);
//                    if (monitor.isCanceled()) {
//                        returnStatus = Status.CANCEL_STATUS;
//                    } else {
//                        monitor.done();
//                    }
//                } catch (final Exception e) {
//                    returnStatus = Status.CANCEL_STATUS;
//                    handleCopyFileException(e);
//                }
//                return returnStatus;
//            }
//        };
//        job.setPriority(Job.SHORT);
//        prepareToRun(job);
//        job.setUser(true);
//        job.schedule();
        
//        Runnable task = new Runnable(){
//            @Override
//            public void run() {
                logger.debug("Marshalling " + logFormat.getName() + " log files, please wait");
                try {
                    logFileMgr.stageLogFiles(raFiles, BaseLogWorker.this);
                } catch (Exception e) {
                    logger.error(e.getMessage(),e);
                }
//            }
//        };
//        TaskScheduler.add(task);
    }

    public void parseFilesIntoTable() {
        if (!logWorkerType.equals("FILE")) {
            return ;
        }
//        Runnable task = new Runnable() {
//            @Override
//            public void run() {
                working = true;
                logger.debug("Parsing " + logFormat.getName() + " log files");
                List<RAFile> fileList = logFormat.getRandAccFileList();
                if (fileList == null /*|| !refreshLock.tryLock()*/) {
                    working = false;
                    return;
                }
                try {
                    LogParser logParser = new LogParser();
                    logParser.loadFromFiles(logFormat, BaseLogWorker.this);

                    // don't call parseFinished() here. It will be
                    // called from the LogParser we just started
                    // when all its jobs are done.
                    // TODO (tproenca): Review this
                } catch (final Exception e) {
                    logger.error(e.getMessage(),e);
//                    parseFinished();
                } finally {
//                    refreshLock.unlock();
                    working = false;
                }
//            }
//        };
//        return TaskScheduler.add(task);
        
        
//        final BaseLogWorker jobLogWorker = this;
//        final LogFormat jobLogFormat = logFormat;
//        final String jobname = "Parsing " + logFormat.getName() + " log files";
//        Job job = new Job(jobname) {
//            protected IStatus run(IProgressMonitor monitor) {
//                IStatus returnStatus = Status.OK_STATUS;
//                LogParser logParser = null;
//                List<RAFile> fileList = jobLogFormat.getRandAccFileList();
//                if (fileList == null) {
//                    parseFinished();
//                    return returnStatus;
//                }
//
//                // only one thread can refresh the table at a time
//                if (!refreshLock.tryLock()) {
//                    parseFinished();
//                    return Status.CANCEL_STATUS;
//                }
//
//                try {
//                    int nFiles = fileList.size();
//                    monitor.beginTask(jobname, nFiles * 10); // report granularity is 10x number of file
//                    logParser = new LogParser();
//                    logParser.loadFromFiles(jobLogFormat, jobLogWorker, monitor);
//                    if (monitor.isCanceled()) {
//                        returnStatus = Status.CANCEL_STATUS;
//                    } else {
//                        monitor.done();
//                    }
//                    // don't call parseFinished() here. It will be
//                    // called from the LogParser we just started
//                    // when all its jobs are done.
//                } catch (final Exception e) {
//                    returnStatus = Status.CANCEL_STATUS;
//                    handleReadFileException(e);
//                    parseFinished();
//                } finally {
//                    refreshLock.unlock();
//                }
//                return returnStatus;
//            }
//        };
//        //job.setPriority(Job.SHORT); // disabled b/c Job.SHORT does not show job progress immediately
//        prepareToRun(job);
//        job.setUser(false);
//        synchronized(this) {
//            working = true;
//        }
//        job.schedule();
    }

    public synchronized boolean isWorking() {
            return working;
    }

    /**
     * The final job in the parser is about to exit,
     * meaning that all files have been parsed and all their
     * data has been added to the filterList.
     */

    public synchronized void parseFinished2() {
            working = false;
    }

    public void loadLogItems() {
        filterList.populateFilteredListWithAllItems();
    }

    public FilterList getFilterList() {
        return filterList;
    }

    public void loadLogItem(LogItem logItem) {
        filterList.add(logItem);
    }

    public ILogItemList getAllItems() {
        return filterList.getAllItems();
    }

    public void setEventList(FilterList eventList) {
        this.filterList = eventList;
    }

    public void setLogFormat(LogFormat logFormat) {
        this.logFormat = logFormat;
    }

    public void setLogWorkerType(String logWorkerType) {
        this.logWorkerType = logWorkerType;
    }

//    public void setRefreshLock(ReentrantLock refreshLock) {
//        this.refreshLock = refreshLock;
//    }

    public void setLogFileMgr(LogFileManager logFileMgr) {
        this.logFileMgr = logFileMgr;
    }

}
