/***********************************************************

 * Copyright 2009 VMware, Inc.  All rights reserved.

 * -- VMware Confidential

 ***********************************************************/



package com.vmware.vide.vlogbrowser.core.fileops;



import java.io.File;

import java.io.FileFilter;

import java.io.FileInputStream;

import java.io.IOException;

import java.text.DateFormat;

import java.text.MessageFormat;

import java.text.SimpleDateFormat;

import java.util.ArrayList;

import java.util.Collections;

import java.util.Comparator;

import java.util.Date;

import java.util.Enumeration;

import java.util.HashSet;

import java.util.List;

import java.util.Set;

import java.util.regex.Matcher;

import java.util.regex.Pattern;

import java.util.zip.GZIPInputStream;



import org.apache.tools.tar.TarEntry;

import org.apache.tools.tar.TarInputStream;

import org.apache.tools.zip.ZipEntry;

import org.apache.tools.zip.ZipFile;

import org.slf4j.Logger;

import org.slf4j.LoggerFactory;



import com.vmware.vide.vlogbrowser.core.consts.LogBrowserConsts;

import com.vmware.vide.vlogbrowser.core.model.LogSystemType;

import com.vmware.vide.vlogbrowser.core.model.LogSystemType.SortMethod;

import com.vmware.vide.vlogbrowser.core.parser.ConfigParser;

import com.vmware.vide.vlogbrowser.core.parser.ManifestFile;

import com.vmware.vide.vlogbrowser.core.parser.RAFile;

import com.vmware.vide.vlogbrowser.core.parser.SolFileParser;

import com.vmware.vide.vlogbrowser.core.utils.ArchiveUtils;



/*

 * This class manages the log files for vLogBrowser. Log content can come from a number

 * of different sources, including:

 * * local log files

 * * local log files that are individually compressed

 * * local log files that are stored together in a compressed archive

 * * remote log files

 * * remote log files that are individually compressed

 * * remote log files whose names match a regular expression

 * * a live log file, fed by command output from an active terminal session

 *

 * Each of these possibilities are described in the .logx configuration file in a standard way.

 * When the vLogBrowser starts up, it reads the .logx configuration file and creates a dynamic

 * list of log files in sorted order. It then copies each log file (RAFile) to a temporary staging

 * directory for use by the log parsing engine. In addition, a manifest can be created,

 * allowing subsequent invocations of vLogBrowser to re-use a set of log files (raFiles) that

 * have already been staged (copied) to the temporary directory.

 */

public class LogFileManager {



    protected ConfigParser cfgParser;

    protected File stagingDirectory;

    protected ManifestFile mfstFile;



    public static final String LOCALHOST = "Local";

    public static final String LOCALHOSTURL = "localhost";  // host name as it appears in URLs



    public static final String SHELL_EXEC = "/bin/sh";

    public static final String SHELL_CMD_SWITCH = "-c";

    public static final String SHELL_LS_CMD = "ls";

    public static final String SHELL_TAR_CMD = "tar";



    public static final String GNU_LS_PARAMS = " -l --time-style=long-iso ";

    public static final int GNU_LS_LINE_REGEX_FILENAME_GROUP = 3;

    public static final String GNU_LS_LINE_REGEX =

        "[-rwx]+\\s+\\d+\\s+\\S+\\s+\\S+\\s+(\\d+)\\s(\\d\\d\\d\\d-\\d\\d-\\d\\d\\s\\d\\d:\\d\\d)\\s";

    public static final String LONG_ISO_TIMESTAMP = "yyyy-MM-dd HH:mm";



    public static final String BUSYBOX_LS_PARAMS = " -lu -e ";

    public static final int BUSYBOX_LS_LINE_REGEX_FILENAME_GROUP = 3;

    public static final String BUSYBOX_LS_LINE_REGEX =

        "[-rwx]+\\s+\\d+\\s+\\S+\\s+\\S+\\s+(\\d+)\\s(\\w\\w\\w\\s\\w\\w\\w\\s+\\d+\\s\\d\\d:\\d\\d:\\d\\d\\s\\d\\d\\d\\d)\\s";

    public static final String BUSYBOX_LS_TIMESTAMP = "EEE MMM d HH:mm:ss yyyy";



    public static final String TGZ_LIST_PARAMS = " -tzvf ";

    public static final String TGZ_EXTRACT_PARAMS = " -xzf ";

    public static final String TGZ_CHANGE_DIR_PARAM = " -C ";

    public static final int TAR_LINE_REGEX_FILENAME_GROUP = 4;

    public static final String TAR_LINE_REGEX =

        "[-rwx]+\\s+\\S+\\s+(\\d+)\\s(\\d\\d\\d\\d-\\d\\d-\\d\\d\\s\\d\\d:\\d\\d(:\\d\\d)*)\\s";

    public static final String TAR_TIMESTAMP = "yyyy-MM-dd HH:mm:ss";



    private static final Logger logger = LoggerFactory.getLogger(LogFileManager.class);

    

    public LogFileManager(ConfigParser cfgParser) throws Exception {

        this.cfgParser = cfgParser;

        String cfgFilePathID = cfgParser.getUniqueID();

        // Get the staging directory, where the manifest file will be

        stagingDirectory = new File(LogBrowserConsts.CACHE_DIR, cfgFilePathID);

    }



// unused:

//    public List<RAFile> marshalLogFiles() throws Exception {

//        List<RAFile> raFiles = null;

//        raFiles = loadFilesFromManifest();

//        if (raFiles != null) {

//            return raFiles;

//        }

//

//        raFiles = getFileList();

//

//        prepareStagingDir(false);

//        // Check if live debugging has been configured, and if so, for what file

//        //  If live debugging configured, start a remote session and start dumping output to the

//        //  configured file

//        if (cfgParser.getLiveLogFileID().length() > 0) {

//            String[] liveLogFields = cfgParser.getLiveLogFileID().split(":");

//            liveLogFile = startLiveLog(liveLogFields[0], liveLogFields[1]);

//        }

//        stageLogFiles(raFiles, new NullProgressMonitor(), null);

//        return raFiles;

//    }



    public File getStagingDirectory() {

        return stagingDirectory;

    }



    public static boolean isLocalHost(String host) {

        if (host.equals(LOCALHOST) || host.equalsIgnoreCase(LOCALHOSTURL))

            return true;

        return false;

    }



    /**

     * Take a List<RAFile>, find each source file, copy that file to the staging area. Whether

     * each source log file is located on a remote ESX / ESXi host, or on the local VM's file

     * tree, it is copied to a temporary location. The log file copy at that location becomes the

     * "target" log file, and each target file is the file actually pointed to by the

     * vLogBrowser file pointers and offsets. If the log file is stored inside a vm-support

     * archive, that archive must first be partially unpacked (i.e. expand the directory that

     * stores the log file(s)) before the log file can be copied (or downloaded) to the

     * target location (which on Linux is in /opt/vmware/vide/vLogBrowser/cache/<logx_ID>/<source_dir_hash>

     *

     * @param raFiles The List<RAFile> that was created with getFileList()

     * @param myMonitor A Progress Monitor

     * @param logWorker The calling LogWorker object, which must be checked for cancellation

     * @throws Exception

     */

    public void stageLogFiles(List<RAFile> raFiles,

                              BaseLogWorker logWorker) throws Exception {

        List<String> remoteCommandsRun = new ArrayList<String>();

        Set<TempUnpackDir> unpackDirs = new HashSet<TempUnpackDir>();

//TODO (tproenca): removed all commented lines

        // For each file in the list under a remote source, copy that file to the local

        //  directory <temp_dir>/<config_file>/<remote_source>

//        boolean isCanceled = false;

        for (RAFile raFile : raFiles) {

            try {

//                if ( (logWorker != null && (logWorker.areAllJobsCanceled())))

//                {

//                    isCanceled = true;

//                    raFile.setCanceled(true); // notify waiting jobs

//                    continue;

//                }

                // prepare to create the target directory and target files

                String parentDir = LogFileUtils.getParentDir(raFile.getSrcFilePath());



                String hashTargetDir =

                    LogFileUtils.computeMD5Hash(raFile.getSrcFileHost() + parentDir);



                String targetDirPath = stagingDirectory.getAbsolutePath() +

                        File.separator + hashTargetDir;

                File targetDir = new File(targetDirPath);

                if (!targetDir.exists()) {

                    targetDir.mkdirs();

                }

                File targetFile = new File(targetDirPath + File.separator +

                        raFile.getSrcFileName(false));

                String srcLogFilePath = raFile.getSrcFilePath();

                String srcArchivePath = raFile.getSrcArchivePath();



                // Separate commands needed for localhost vs. remote host through RSE

                if (isLocalHost(raFile.getSrcFileHost())) {

                    if ((srcArchivePath != null) && (! srcArchivePath.isEmpty())) {

                        LogFileUtils.extractFileFromArchive(srcArchivePath, srcLogFilePath, targetFile,

                                cfgParser.getUniqueID(), unpackDirs, raFile);

                    } else {

                        LogFileUtils.localFileCopy(new File(srcLogFilePath), targetFile);

                    }

                } else {

//                    ITarget remoteHost = LogFileUtils.getTarget(raFile.getSrcFileHost());

//                    // Check to see if the log file is in an archive...

//                    if ((srcArchivePath != null) && (! srcArchivePath.isEmpty())) {

//                        TempUnpackDir tempUnpackDir = new TempUnpackDir(cfgParser.getUniqueID(), raFile);

//                        // run remote command

//                        String remoteCommand = SHELL_EXEC + " " + SHELL_CMD_SWITCH + " '" +

//                            tempUnpackDir.getRemoteCreateCmd() + "; " + // Run both mkdir -p and tar -xvf in one command

//                            SHELL_TAR_CMD + TGZ_EXTRACT_PARAMS + LogFileUtils.escapeSpaces(srcArchivePath) +

//                            TGZ_CHANGE_DIR_PARAM + tempUnpackDir.toString() +

//                            " " + LogFileUtils.escapeSpaces(LogFileUtils.getParentDir(raFile.getSrcFilePath(), "/")) + "'";

//                        if (!remoteCommandsRun.contains(remoteCommand)) {

//                            LogFileUtils.executeCommand(remoteHost, remoteCommand);

//                            remoteCommandsRun.add(remoteCommand);

//                            unpackDirs.add(tempUnpackDir);

//                        }

//                        srcLogFilePath = tempUnpackDir.toString() + srcLogFilePath;

//                    }

//                    LogFileUtils.downloadFile(remoteHost,

//                                              srcLogFilePath,

//                                              targetDirPath);

                }

                // If the file (i.e. the "target" log file the temporary location) is compressed,

                // decompress it

                if (targetFile.exists()) {

                    if (targetFile.getName().endsWith(".gz")) {

                        int gzStrIdx = raFile.getSrcFileName(false).indexOf(".gz");

                        String gunzipName = raFile.getSrcFileName(false).substring(0, gzStrIdx);

                        String gunzipPath = targetDirPath + File.separator + gunzipName;

                        try {

                            LogFileUtils.decompressGzipFile(targetFile, gunzipPath);

                        }

                        catch (IOException ex) {

                            // re-throw exception with filename in the message

                            String message = ex.getMessage() + ":\n" + targetFile;

                            throw new IOException(message, ex);

                        }

                        LogFileUtils.deleteDirTree(targetFile);

                        targetFile = new File(gunzipPath);

                    } else if (targetFile.getName().endsWith(".sol")) {

                        String name = raFile.getSrcFileName(false);

                        name = name.substring(0, name.lastIndexOf('.')); // remove extension

                        String solLogFileName = name + ".log";

                        String solLogFilePath = targetDirPath + File.separator + solLogFileName;

//                        try {

                            SolFileParser.decode(targetFile, new File(solLogFilePath));

//                        } catch (Exception e) {

                            // re-throw exception with filename in the message

//                            String message = e.getMessage() + ":\n" + targetFile;

//                            throw new Exception(message, e);

//                        }

                        LogFileUtils.deleteDirTree(targetFile);

                        targetFile = new File(solLogFilePath);

                    }

                    raFile.setTargetFilePath(targetFile.getAbsolutePath());

                } else {

                    // file not found after attempted copy--something went wrong

//                    raFile.setCanceled(true); // notify waiting jobs

                    String msg = "Unable to stage a temporary copy of file '{0}' for parsing";

                    msg = MessageFormat.format(msg, raFile.toDisplayString());

                    throw new Exception(msg);

                }

            } catch (Exception e) {

                logger.error("Error staging log files.", e);

                raFile.setCanceled(true); // notify waiting jobs

//                throw new Exception("Problem staging log files: " + e.getMessage());

            }

//            myMonitor.worked(1);



        } // for raFiles



        // Clean up temporary directory used for unpacking tar file

        TempUnpackDir[] dirs = unpackDirs.toArray(new TempUnpackDir[0]);

        for (TempUnpackDir dir : dirs) {

            if (isLocalHost(dir.getSourceFileHost())) {

                LogFileUtils.deleteDirTree(new File(dir.toString()));

            }

//            } else {

//                // Remote Location

//                ITarget target = LogFileUtils.getTarget(dir.getSourceFileHost());

//                LogFileUtils.executeCommand(target, "rm -rf " + dir.toString());

//            }

        }



        // check if loading is canceled

//        if (isCanceled)

//            return;



        // Now that a list of target log files is ready in the temporary staging location,

        // Create the manifest file which lists all target log files in this particular

        // temporary location. The next time the vLogBrowser window is opened, it will

        // try to load the log files from this temporary cache instead of

        // recopying or re-downloading them. However, if this temporary cache is deleted

        // or corrupted, normal copy / download procedures run (the above loop) without complaint

        if ((mfstFile == null) || (! ManifestFile.isLegalManifest(mfstFile.getFile()))) {

            mfstFile = new ManifestFile(stagingDirectory.getCanonicalPath(),

                                        cfgParser.getConfigFilePath(),

                                        false);

        }

//        myMonitor.worked(1);

        mfstFile.addRAFiles(raFiles);

    }



    public void prepareStagingDir(boolean forceReload) throws Exception {

        if (forceReload) {

            // This will delete dir if already present

            LogFileUtils.deleteDirTree(stagingDirectory);

            if (stagingDirectory.exists()) {

                throw new Exception("Cannot remove old temp directory to make room for new" +

                        " one: '" + stagingDirectory.getAbsolutePath() + "'.");

            }

        }



        stagingDirectory.mkdirs();

        if (!stagingDirectory.exists()) {

            throw new Exception("Cannot create temp directory '"

                    + stagingDirectory.getAbsolutePath() + "'.");

        }

    }



    public List<RAFile> loadFilesFromManifest() throws Exception {

        mfstFile = new ManifestFile(stagingDirectory.getCanonicalPath(),

                                    cfgParser.getConfigFilePath(),

                                    false);

        if (! mfstFile.isValidFileList()) {

            mfstFile.initNewManifestFile(cfgParser.getConfigFilePath());

        }

        return mfstFile.getFileList();

    }



    /**

     * Retrieve a List<RAFile>, i.e. a list of log files that should be loaded into vLogBrowser.

     * This list is then sent to stageLogFiles() above, such that the individual log files

     * can be copied locally or downloaded from the remote location.

     * @return

     * @throws Exception

     */

    public List<RAFile> getFileList() throws Exception {

        List<RAFile> raFiles = new ArrayList<RAFile>();

        String[] sourceIDs = cfgParser.getSourceIDs();

        for (String srcID : sourceIDs) {

            String[] srcIDparts = srcID.split(";");

            String host = srcIDparts[0];    // first string in ID is source host

            String archiveFilepath = null;

            if (srcIDparts.length > 2) {    // archive file is in middle of ID, if present

                archiveFilepath = srcIDparts[srcIDparts.length - 2];

            }

            String parentDir = srcIDparts[srcIDparts.length - 1];  // last string is parentDir

            String[] fileRegExArr = cfgParser.getFileRegExesInSrc(srcID);

            LogSystemType systemType = cfgParser.getSystemType();

            boolean fileSeqNumIsDescending = systemType.isFileNumDescending();

            for (String fileRegEx : fileRegExArr) {

                // support multiple locations (e.g. tgz vs. live system)

                String[] paths = parentDir.split(Pattern.quote("|"));

                for (String path : paths) {

                    List<RAFile> fileList = getMatchingFiles(host,

                                                         archiveFilepath,

                                                         path.trim(),

                                                         fileRegEx,

                                                         fileSeqNumIsDescending);

                    if (fileList.size() > 0) {

                        // only one location to avoid duplicates

                        raFiles.addAll(fileList);

                        break;

                    }

                }

            }



            // sort filenames according to sort method specified in template declaration

            if (systemType.getSortMethod() == SortMethod.Time) {

                // sort by filetime

                Comparator<RAFile> comp = new Comparator<RAFile>() {

                    @Override

                    public int compare(RAFile f1, RAFile f2) {

                        Date date1 = f1.getSrcFileTimestamp();

                        Date date2 = f2.getSrcFileTimestamp();

                        int comp = date1.compareTo(date2);

                        return comp;

                    }

                };

                Collections.sort(raFiles, comp);

            }

            else if (systemType.getSortMethod() == SortMethod.Filename) {

                // sort by filename

                Comparator<RAFile> comp = new Comparator<RAFile>() {

                    @Override

                    public int compare(RAFile f1, RAFile f2) {

                        String name1 = f1.getSrcFileName(false);

                        String name2 = f2.getSrcFileName(false);

                        int comp = name1.compareTo(name2);

                        return comp;

                    }

                };

                Collections.sort(raFiles, comp);

            }

            else {

                // sort by filenum

                Collections.sort(raFiles);

            }

        }



        return raFiles;

    }



    /**

     * Heuristic to reduce the size of the file list to be processed

     * based on a specified limit.

     * This is used to avoid running out of heap space and/or disk

     * space for very large log file sets, such as VC support bundles.

     * Since the (compressed) bundle may contain compressed log files

     * we cannot calculate a precise size for the expanded log files

     * unless we expand the bundle, which would defy the purpose of limiting

     * the size.

     * Therefore, we assume that any compressed log file in the bundle

     * (file name ending with .gz) will be expanded 10 times, and use

     * that to calculate the final size.

     * We return the list of files (from the end of the original list,

     * i.e. most recent entries) that fits within the specified size limit.

     *

     * @param fileList unlimited list of log files to process

     * @param limitMB size limit in MB of the resulting list of log files

     * @return the trimmed list of log files

     */



    public List<RAFile> trimFileList(List<RAFile> fileList, int limitMB) {

        long    size = 0;

        int     indx;

        long    limit = limitMB;

        limit *= 1024 * 1024;           // make it into bytes

        for (indx = fileList.size() - 1; indx >= 0; --indx) {

            RAFile f = fileList.get(indx);

            if (f.getSrcFileName(false).endsWith(".gz"))

                size += f.getSizeInBytes() * 10;

            else

                size += f.getSizeInBytes();

            if (size > limit) {

                return fileList.subList(indx, fileList.size());

            }

        }

        return fileList;

    }



    /**

     * Return the list of files present in a ZIP archive

     * in a string format that is compatible with tar's output.

     * This way other parts of the code don't need to change.

     *

     * @param archive

     * @return

     */



    private static CommandData getTarFileListFromZip(File archive) {

        CommandData cmdResult = new CommandData();

        ZipFile zFile = null;

        StringBuilder sb = new StringBuilder();

        try {

            zFile = new ZipFile(archive);

            for (Enumeration<?> e = zFile.getEntries(); e.hasMoreElements();) {

                ZipEntry ze = (ZipEntry) e.nextElement();

                String name = ze.getName();

                // build a Tar-like representation of the file entry

                sb.append("-r--r--r-- root/root  ");

                sb.append(ze.getSize());

                sb.append(" 2011-11-11 00:00:00 ");

                sb.append(name);

                sb.append("\n");

            }

        } catch (IOException e) {

            e.printStackTrace();

        } finally {

            if (zFile != null) {

                try {

                    zFile.close();

                } catch (IOException e) {

                    e.printStackTrace();

                }

            }

        }

        cmdResult.setOutput(sb.toString());

        return cmdResult;

    }



    /**

     * Retrieve a list of all the files that match a single source ID

     * in the .logx configuration file.

     * @param host

     * @param archiveFilepath

     * @param parentDir

     * @param fileRegEx

     * @param fileSeqNumIsDescending

     * @return

     * @throws Exception

     */

    public static List<RAFile> getMatchingFiles(String host,

                                         String archiveFilepath,

                                         String parentDir,

                                         String fileRegEx,

                                         boolean fileSeqNumIsDescending) throws Exception {

        CommandData cmdResult = null;

        boolean isRemoteESXi = false;

        // Different commands and parameters are necessary for locally stored log files,

        // remote log files on an ESX server, remote log files on an ESXi server, log files

        // stored in a .tgz (vm-support) archive. Each of these cases is separate and handled

        // by the conditionals below.

//        ITarget remoteTarget = LogFileUtils.getTarget(host);

        if (isLocalHost(host) /*||

                (remoteTarget != null &&

                        ITargetManager.TARGET_TYPE_LOCAL.equals(remoteTarget.getTargetType()))*/) {

            if (archiveFilepath == null) {      // log file is not stored in a .tgz archive

                return getLocalMatchingFiles(parentDir, fileRegEx, fileSeqNumIsDescending);

            }

            File archiveFile = new File(archiveFilepath);

            if (ArchiveUtils.isGZipTarFile(archiveFile)) {

                //if (Utils.isPlatformWindows()) {

                    // FIXME: The Apache TAR library does not handle vm-support logs that contain

                    // core files very well. It chokes on the core file entries within the tar file.

                    // Workaround: avoid using tar files that contain core files.

                    return getLocalFilesFromTar(archiveFilepath, parentDir, fileRegEx, fileSeqNumIsDescending);

                //}

                // For Linux systems, there is an alternative to the Apache TAR library:

                // direct use of the tar command in the shell. When Apache TAR library handles

                // vm-support logs (that contain core files) properly, we can use getLocalFilesFromTar()

                // for both Windows and Linux, and eliminate the following three statements

                /*String commandParam = SHELL_TAR_CMD + TGZ_LIST_PARAMS +

                    LogFileUtils.escapeSpaces(archiveFilepath);

                String[] localCommand = new String[]{SHELL_EXEC, SHELL_CMD_SWITCH, commandParam};

                cmdResult = LogFileUtils.runLocalCommand(localCommand, "/");*/

            } else if (ArchiveUtils.isZipFile(archiveFile)) {

                cmdResult = getTarFileListFromZip(archiveFile);

            }

        } /*else {

            if (remoteTarget == null) {

                throw new IOException("host " + host + " is not available");

            }



            if (archiveFilepath == null) {

                List<RAFile> fileList = new ArrayList<RAFile>();

                Pattern pt = Pattern.compile(fileRegEx + "\\S*$");

                if (!remoteTarget.isConnected()) {

                    remoteTarget.connect(null);

                }

                ITargetFile tf = remoteTarget.getTargetFile(parentDir, null);

                char separator = remoteTarget.getTargetType() == ITargetManager.TARGET_TYPE_WIN ? '\\' : '/';

                ITargetFile[] filelist = tf.listFiles(null);

                for (ITargetFile f : filelist) {

                    IPath p = new RemotePath(f);

                    Matcher m = pt.matcher(p.lastSegment());

                    if (m.matches()) {

                        RAFile newRAFile = new RAFile();

                        Date date = new Date(f.lastModified());

                        newRAFile.setSrcFileTimestamp(date);

                        StringBuilder sb = new StringBuilder(parentDir);

                        sb.append(p.lastSegment());

                        newRAFile.setSrcFilePath(sb.toString());

                        newRAFile.setSrcFileHost(host);

                        newRAFile.setSrcSeparator(separator);

                        newRAFile.setSizeInBytes((int) f.length());

                        newRAFile.setSeqNumInvertedForSort(fileSeqNumIsDescending);

                        newRAFile.setSrcArchivePath(archiveFilepath);

                        fileList.add(newRAFile);

                    }

                }

                return fileList;

            }



            if (remoteTarget.isSupportSsh()) {

                isRemoteESXi = remoteTarget.isVisor();

                if (archiveFilepath.endsWith("tgz")) {

                    String remoteCommand = SHELL_EXEC + " " + SHELL_CMD_SWITCH + " '" +

                        SHELL_TAR_CMD + TGZ_LIST_PARAMS +

                            LogFileUtils.escapeSpaces(archiveFilepath) + "'";

                    cmdResult = LogFileUtils.executeCommand(remoteTarget, remoteCommand);

                }

            }



        }*/



        // For cases not covered by the conditionals (e.g. unsupported remote host types),

        // this method returns an empty list.

        if ((cmdResult != null) && (cmdResult.getReturnCode() == 0)) {

            return parseFileList(host,

                                 archiveFilepath,

                                 parentDir,

                                 fileRegEx,

                                 cmdResult.getOutput(),

                                 fileSeqNumIsDescending,

                                 isRemoteESXi);

        } else {

            return new ArrayList<RAFile>();

        }

    }



    private static List<RAFile> getLocalMatchingFiles(String parentDir,

                                                      String fileRegEx,

                                                      boolean fileSeqNumIsDescending) {

        List<RAFile> raFiles = new ArrayList<RAFile>();

        File[] fileArr = new File[0];

        final Pattern namePattern = Pattern.compile(fileRegEx);

        try {

            if (parentDir != null) {

                File file = new File(parentDir);

                if (file.exists() && file.isDirectory()) {

                    fileArr = file.listFiles(new FileFilter() {

                        @Override

                        public boolean accept(File path) {

                            Matcher myMatcher = namePattern.matcher(path.getName());

                            return path.isFile() && myMatcher.find();

                        }

                    });

                }

            }

        } catch (Exception e) {

            e.printStackTrace();

        }

        for(File file : fileArr) {

            RAFile newRAFile = new RAFile();

            newRAFile.setSrcFileTimestamp(new Date(file.lastModified()));

            newRAFile.setSrcFilePath(file.getAbsolutePath());

            newRAFile.setSrcFileHost(LOCALHOST);

            newRAFile.setSrcSeparator(File.separatorChar);

            newRAFile.setSizeInBytes((int)file.length());

            newRAFile.setSeqNumInvertedForSort(fileSeqNumIsDescending);

            raFiles.add(newRAFile);

        }

        return raFiles;

    }



    private static List<RAFile> getLocalFilesFromTar(String archiveFilepath,

                                                     String parentDir,

                                                     String fileRegEx,

                                                     boolean fileSeqNumIsDescending) throws Exception {

        List<RAFile> raFiles = new ArrayList<RAFile>();

        String quotedParentDir = Pattern.quote(parentDir);

        Pattern dirFilePattern = Pattern.compile(quotedParentDir + fileRegEx);

        GZIPInputStream gis = new GZIPInputStream(new FileInputStream(new File(archiveFilepath)));

        TarInputStream tarInStrm = new TarInputStream(gis);

        TarEntry tarEntry = tarInStrm.getNextEntry();

        while (tarEntry != null) {

            if (! tarEntry.isDirectory()) {

                Matcher dirFileMatcher = dirFilePattern.matcher(tarEntry.getName());

                if (dirFileMatcher.find()) {

                    RAFile newRAFile = new RAFile();

                    newRAFile.setSrcFileTimestamp(tarEntry.getModTime());

                    newRAFile.setSrcFilePath(tarEntry.getName());

                    newRAFile.setSrcFileHost(LOCALHOST);

                    newRAFile.setSrcSeparator(File.separatorChar);

                    newRAFile.setSizeInBytes((int)tarEntry.getSize());

                    newRAFile.setSeqNumInvertedForSort(fileSeqNumIsDescending);

                    newRAFile.setSrcArchivePath(archiveFilepath);

                    raFiles.add(newRAFile);

                }

            }

            try {

                // FIXME: This fails on certain vm-support .tgz files--specifically those that

                // contain core files. This is either because the Apache TAR library has a problem,

                // or VMware's vm-support tar format is not correct.

                tarEntry = tarInStrm.getNextEntry();

            } catch (IOException ioe) {

                System.err.println("Caught an I/O exception: " + ioe.getMessage());

                throw new Exception("Tar file entry unreadable. Try re-creating the vm-support log" +

                        " without core files using 'vm-support -n'... " + ioe.getMessage());

            }

        }

        tarInStrm.close();

        return raFiles;

    }



    /**

     * Take the command output (which is either a directory listing or a tar listing) and

     * get a list of log files that match the fileRegEx. Also get information, like file

     * dates and sizes. Each log file referenced in the List<RAFile> will be augmented with

     * this information.

     * @param host

     * @param archiveFilepath

     * @param parentDir

     * @param fileRegEx

     * @param fileListStr

     * @param fileSeqNumIsDescending

     * @param isRemoteESXi

     * @return

     * @throws Exception

     */

    public static List<RAFile> parseFileList(String host,

                                      String archiveFilepath,

                                      String parentDir,

                                      String fileRegEx,

                                      String fileListStr,

                                      boolean fileSeqNumIsDescending,

                                      boolean isRemoteESXi) throws Exception {

        List<RAFile> fileList = new ArrayList<RAFile>();

        String lineRegexStr = null;

        DateFormat dateFormat = null;

        int filenameGroup = 0;

        // Prepare the appropriate regular expression, depending on the source log location

        if (archiveFilepath != null && (archiveFilepath.endsWith(".tgz") ||

                archiveFilepath.endsWith(".zip"))) {

            lineRegexStr = "^" + TAR_LINE_REGEX + "([^/]*" + parentDir + fileRegEx + "\\S*)$";

            dateFormat = new SimpleDateFormat(TAR_TIMESTAMP);

            filenameGroup = TAR_LINE_REGEX_FILENAME_GROUP;

        } else if (isRemoteESXi) {

            lineRegexStr = "^" + BUSYBOX_LS_LINE_REGEX + "(" + fileRegEx + "\\S*)$";

            dateFormat = new SimpleDateFormat(BUSYBOX_LS_TIMESTAMP);

            filenameGroup = BUSYBOX_LS_LINE_REGEX_FILENAME_GROUP;

        } else {

            lineRegexStr = "^" + GNU_LS_LINE_REGEX + "(" + fileRegEx + "\\S*)$"; // ESX Classic

            dateFormat = new SimpleDateFormat(LONG_ISO_TIMESTAMP);

            filenameGroup = GNU_LS_LINE_REGEX_FILENAME_GROUP;

        }

        Pattern lineRegexPattern = Pattern.compile(lineRegexStr);

        String[] lines = fileListStr.split("\n");

        // Try to match each line in the file list output to a log file RegEx, and retrieve

        // the other file information elements (date, size) as well.

        for (String line : lines) {

            Matcher lineMatcher = lineRegexPattern.matcher(line);

            String sizeValueStr = null;

            String dateValueStr = null;

            String fileNameValueStr = null;

            Date date = null;

            if (lineMatcher.matches()) {

                try {

                    sizeValueStr = line.substring(lineMatcher.start(1), lineMatcher.end(1));

                    dateValueStr = line.substring(lineMatcher.start(2), lineMatcher.end(2));

                    fileNameValueStr = line.substring(lineMatcher.start(filenameGroup), lineMatcher.end(filenameGroup));

                    if (archiveFilepath == null) {

                        fileNameValueStr = parentDir + fileNameValueStr;

                    }



                    // try to parse date with current and iso timestamp

                    // (ubuntu tar timestamp may not have seconds)

                    try {

                        date = dateFormat.parse(dateValueStr);

                    }

                    catch (Exception ex) {

                        dateFormat = new SimpleDateFormat(LONG_ISO_TIMESTAMP);

                        date = dateFormat.parse(dateValueStr);

                    }

                    RAFile newRAFile = new RAFile();

                    newRAFile.setSrcFileTimestamp(date);

                    newRAFile.setSrcFilePath(fileNameValueStr);

                    newRAFile.setSrcFileHost(host);

                    /* Assume that support only unix type file system. */

                    newRAFile.setSrcSeparator('/');

                    newRAFile.setSizeInBytes(Integer.parseInt(sizeValueStr));

                    newRAFile.setSeqNumInvertedForSort(fileSeqNumIsDescending);

                    newRAFile.setSrcArchivePath(archiveFilepath);

                    fileList.add(newRAFile);

                } catch (Exception e) {

                    e.printStackTrace();

                    continue;

                }

            }

        }

        return fileList;

    }



    public ConfigParser getCfgParser() {

        return cfgParser;

    }



}

