/***********************************************************

 * Copyright 2009 VMware, Inc.  All rights reserved.

 * -- VMware Confidential

 ***********************************************************/



package com.vmware.vide.vlogbrowser.core.fileops;



import java.io.BufferedInputStream;

import java.io.BufferedOutputStream;

import java.io.BufferedReader;

import java.io.File;

import java.io.FileInputStream;

import java.io.FileOutputStream;

import java.io.IOException;

import java.io.InputStream;

import java.io.InputStreamReader;

import java.io.OutputStream;

import java.math.BigInteger;

import java.nio.channels.FileChannel;

import java.security.MessageDigest;

import java.util.Date;

import java.util.List;

import java.util.Set;

import java.util.regex.Matcher;

import java.util.regex.Pattern;

import java.util.zip.GZIPInputStream;

import java.util.zip.ZipEntry;

import java.util.zip.ZipException;

import java.util.zip.ZipFile;



import org.apache.commons.compress.archivers.tar.TarArchiveEntry;

import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;

import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;



import com.vmware.vide.vlogbrowser.core.parser.RAFile;

import com.vmware.vide.vlogbrowser.core.utils.ArchiveUtils;



/*

 * This class contains a number of helper methods that can be used to transfer and manipulate

 * log files stored in local or remote locations.

 */

public class LogFileUtils {



    public static final int ZIP_BUFFER_SIZE = 512;



    private static final int BUFFER_SIZE = 4 * 1024; // 4KB



//    private static class WaitForRSEInit implements ITargetManagerInitListener {

//

//        boolean initComplete;

//

//        public WaitForRSEInit(boolean alreadyInitialized) {

//            if (alreadyInitialized) {

//                initComplete = true;

//            } else {

//                initComplete = false;

//            }

//        }

//

//        @Override

//        public synchronized void phaseComplete(int phase) {

//            if (phase == ITargetManagerInitListener.INIT_PHASE_ALL) {

//                initComplete = true;

//            }

//            notifyAll();

//        }

//

//        public synchronized void waitForInit() throws Exception {

//            while (!initComplete) {

//                try {

//                    wait();

//                } catch (InterruptedException e) {

//                    e.printStackTrace();

//                }

//            }

//            return;

//        }

//

//    }



//    /**

//     * Upload a file to a remote target

//     *

//     * @param target

//     * @param filePath

//     * @param destDir

//     * @throws Exception

//     */

//    public static void uploadFile(ITarget target, String filePath, String destDir)

//            throws Exception {

//        target.connect(null);

//        target.uploadFile(new File(filePath), destDir, null);

//    }



    /**

     * Download a file from a remote target

     *

     * @param target

     * @param remoteFilePath

     * @param localDestDir

     * @throws Exception

     */

//    public static void downloadFile(ITarget target, String remoteFilePath, String localDestDir)

//            throws Exception {

//        target.connect(null);

//        target.downloadFile(remoteFilePath, new File(localDestDir), null);

//    }



    /**

     * Execute a command on a remote target

     *

     * @param target

     * @param command

     * @return

     * @throws Exception

     */

//    public static CommandData executeCommand(ITarget target, String command) throws Exception {

//        class OutputListener implements ICommandOutputListener {

//            private int count = 0;

//            private List<String> lines = new ArrayList<String>();

//

//            public synchronized void commandOutputChanged(ICommandOutputChangedEvent event) {

//                String[] output = event.getOutputLines();

//                for (int i = 0; i < output.length; ++i) {

//                    lines.add(output[i]);

//                }

//                count += output.length;

//            }

//

//            public synchronized String[] getLines() {

//                return lines.toArray(new String[lines.size()]);

//            }

//        }

//

//        if (target == null) {

//            throw new Exception("Error: cannot execute command '" + command +

//                    "' because target is null!");

//        }

//        target.connect(null);

//        CommandData output = new CommandData();

//        ICommandOutputListener outputListener = (ICommandOutputListener) new OutputListener();

//        try {

//            int executeCommand = target.executeCommand(command, outputListener, null);

//            output.setReturnCode(executeCommand);

//        } catch (Exception e) {

//            System.out.println("Caught an exception trying to setReturnCode " + e.getMessage());

//            e.printStackTrace();

//        }

//        output.setOutput(arrayToString(((OutputListener) outputListener).getLines(), "\n"));

//        return output;

//    }



    /**

     * Find target using either hostname or IP address

     *

     * @param targetName

     * @return The target with the corresponding hostname, null if the target is not found.

     * @throws Exception

     */

//    public static ITarget getTarget(String targetName) throws Exception {

//        waitForRSEInit();

//        return TargetManagerPlugin.getTheTargetManager().findTarget(targetName);

//    }



    /**

     * @deprecated  // DEBUG need to remove this method when ready, in favor of RSE native method

     * getTargets()

     *

     * @param myMonitor

     * @return

     * @throws Exception

     */

//    public static String[] getAllTargetNames(IProgressMonitor myMonitor) throws Exception {

//        myMonitor.subTask("Initializing Remote System Explorer...");

//        waitForRSEInit();

//        myMonitor.worked(1);

//        myMonitor.subTask("Retrieving remote connection list...");

//        ITarget[] targets = TargetManagerPlugin.getTheTargetManager().getTargets();

//        List<String> targetList = new ArrayList<String>();

//        targetList.add("localhost");

//        for (int i = 0; i < targets.length; i++) {

//            String name = targets[i].getName();

//            if ((!name.equalsIgnoreCase("localhost")) && (!name.equalsIgnoreCase("local"))) {

//                targetList.add(name);

//            }

//        }

//        myMonitor.worked(1);

//        return targetList.toArray(new String[targetList.size()]);

//    }



//    public static List<String> getTargetList(boolean prependLocalhost) throws Exception {

//        ITarget[] targets = TargetManagerPlugin.getTheTargetManager().getTargets();

//        List<String> targetList = new ArrayList<String>();

//        if (prependLocalhost) {

//            targetList.add(LogFileManager.LOCALHOST);

//        }

//        for (int i = 0; i < targets.length; i++) {

//            String name = targets[i].getName();

//            if ((!name.equalsIgnoreCase(LogFileManager.LOCALHOST)) &&

//                (!name.equalsIgnoreCase("local")) ) {

//                targetList.add(name);

//            }

//        }

//        return targetList;

//    }

//

//    private static void waitForRSEInit() throws Exception {

//        WaitForRSEInit waitForIt = new WaitForRSEInit(TargetManagerPlugin.getTheTargetManager()

//                .isInitialized());

//        TargetManagerPlugin.getTheTargetManager().addInitListener(waitForIt);

//

//        if (!TargetManagerPlugin.getTheTargetManager().isInitialized()) {

//            waitForIt.waitForInit();

//        }

//    }

//

//    public static void createESXTarget(String targetName,

//                                       String hostNetName,

//                                       String userID,

//                                       String pswd) throws Exception {

//        if (getTarget(targetName) != null) {

//            return;

//        }

//        String targetType = ITargetManager.TARGET_TYPE_ESX;

//        ITarget myTarget = TargetManagerPlugin.getTheTargetManager().createTarget(

//                targetType, targetName, hostNetName);

//        myTarget.setUserId(userID);

//        myTarget.setPassword(pswd, true);

//    }



    /**

     * Smash a String array into a single String object

     *

     * @author fincht

     * @param stringArray

     * @param delimiter

     * @return

     */

    public static String arrayToString(String[] stringArray, String delimiter) {

        StringBuffer resultBuffer = new StringBuffer();

        if (stringArray.length > 0) {

            resultBuffer.append(stringArray[0]);

            for (int i = 1; i < stringArray.length; i++) {

                resultBuffer.append(delimiter);

                resultBuffer.append(stringArray[i]);

            }

        }

        return resultBuffer.toString();

    }



    public static CommandData runLocalCommand(String[] localCommand, String parentDir) throws Exception {

        CommandData returnData = new CommandData();

        ProcessBuilder processBuilder = new ProcessBuilder(localCommand);

        processBuilder.directory(new File(parentDir));

        Process proc = processBuilder.start();

        InputStream input = proc.getInputStream();

        BufferedReader reader = new BufferedReader(new InputStreamReader(input));

        String line = null;

        StringBuffer resultBuffer = new StringBuffer();

        while ((line = reader.readLine()) != null) {

            resultBuffer.append("\n");

            resultBuffer.append(line);

        }

        returnData.setReturnCode(proc.waitFor());

        returnData.setOutput(resultBuffer.toString());

        return returnData;

    }



    public static boolean deleteDirTree(File directoryTree) {

        if ((directoryTree.isFile()) ||

                ((directoryTree.list() != null) && (directoryTree.list().length == 0))) {

            return directoryTree.delete();

        } else {

            File[] dirContents = directoryTree.listFiles();

            if (dirContents != null) {

                for (File file : dirContents) {

                    if (!deleteDirTree(file)) {

                        return false;

                    }

                }

            }

            return directoryTree.delete();

        }

    }



    public static String computeMD5Hash(String plainText) throws Exception {

        MessageDigest md5 = MessageDigest.getInstance("MD5");

        return new BigInteger(1, md5.digest(plainText.getBytes())).toString(16);

    }



    public static String padZeros(String missingLeadingZeros, int places) {

        String leadingZeros = "";

        for (int i = 0; i < places - missingLeadingZeros.length(); i++) {

            leadingZeros += "0";

        }

        return leadingZeros + missingLeadingZeros;

    }



    public static String concatPath(String path, char glue) {

        String returnStr = new String(path);

        returnStr = returnStr.replace('/', glue);

        returnStr = returnStr.replace('\\', glue);

        returnStr = returnStr.replace(':', glue);

        returnStr = returnStr.replace('.', glue);

        return returnStr;

    }



    public static void localFileCopy(File src, File dest) throws Exception {

        if ((!src.exists()) || (!src.canRead())) {

            throw new Exception("Cannot read from source file");

        }

        if (dest.isDirectory()) {

            deleteDirTree(dest);

        } else {

            dest.delete();

        }

        if (!dest.createNewFile()) {

            throw new Exception("Destination file could not be created or overwritten.");

        }

        FileChannel srcChannel = new FileInputStream(src).getChannel();

        FileChannel destChannel = new FileOutputStream(dest).getChannel();

        try {

            srcChannel.transferTo(0L, srcChannel.size(), destChannel);

        } catch (Exception e) {

            throw new Exception("Failed to copy " + src.getAbsolutePath() + " to " +

                    dest.getAbsolutePath() + " " + e.getMessage());

        } finally {

            if (srcChannel != null) {

                srcChannel.close();

            }

            if (destChannel != null) {

                destChannel.close();

            }

        }

    }



    public static void extractFileFromTgz(String tgzFile, String srcFile,

            File destFile) throws Exception {

        TarArchiveInputStream tais = new TarArchiveInputStream(

                new GzipCompressorInputStream(new FileInputStream(new File(

                        tgzFile))));

        TarArchiveEntry entry = null;

        do {

            entry = tais.getNextTarEntry();

        } while (entry != null && !entry.getName().equals(srcFile));



        if (entry != null) {

            File targetDir = destFile.getParentFile();

            if (targetDir != null) {

                targetDir.mkdirs();

            }

            OutputStream fout = new BufferedOutputStream(new FileOutputStream(

                    destFile));

            byte[] buffer = new byte[BUFFER_SIZE];

            long size = entry.getSize();

            long n = 0;

            while (n < size) {

                int b_read = tais.read(buffer);

                fout.write(buffer, 0, b_read);

                n += b_read;

            }

            fout.close();

            tais.close();

        }

    }



    public static void decompressFileInZipArchive(File archiveFile,

                                                  String internalFilePath,

                                                  File targetFile) throws ZipException,

                                                                                IOException {

        ZipFile zipFile = new ZipFile(archiveFile);

        ZipEntry zipEntry = zipFile.getEntry(internalFilePath);

        InputStream zipInputStrm = zipFile.getInputStream(zipEntry);

        BufferedInputStream buffInputStrm = new BufferedInputStream(zipInputStrm);

        byte[] byteBuffer = new byte[ZIP_BUFFER_SIZE];

        FileOutputStream fileOutputStrm = new FileOutputStream(targetFile);

        BufferedOutputStream buffOutputStrm = new BufferedOutputStream(fileOutputStrm,

                                                                       ZIP_BUFFER_SIZE);

        int count = 0;

        while ((count = buffInputStrm.read(byteBuffer, 0, ZIP_BUFFER_SIZE)) != -1) {

            buffOutputStrm.write(byteBuffer, 0, count);

        }

        buffOutputStrm.flush();

        buffOutputStrm.close();

        buffInputStrm.close();

    }



    public static void decompressGzipFile(File gZipFile,

                                          String outputFilePath) throws IOException {

        FileInputStream fileInputStrm = new FileInputStream(gZipFile);

        GZIPInputStream gZipInputStrm = new GZIPInputStream(fileInputStrm);

        byte[] byteBuffer = new byte[ZIP_BUFFER_SIZE];

        FileOutputStream fileOutputStrm = new FileOutputStream(outputFilePath);

        BufferedOutputStream buffOutputStrm =

            new BufferedOutputStream(fileOutputStrm, ZIP_BUFFER_SIZE);

        int count = 0;

        while ((count = gZipInputStrm.read(byteBuffer, 0, ZIP_BUFFER_SIZE)) != -1) {

            buffOutputStrm.write(byteBuffer, 0, count);

        }

        buffOutputStrm.flush();

        buffOutputStrm.close();

        gZipInputStrm.close();

    }



    public static void extractFileFromArchive(String srcArchivePath,

            String srcLogFilePath, File targetFile, String uniqueId,

            Set<TempUnpackDir> unpackDirs, RAFile raFile) throws Exception {

        File archive = new File(srcArchivePath);

        if (ArchiveUtils.isGZipTarFile(archive)) {

            LogFileUtils.extractFileFromTgz(srcArchivePath, srcLogFilePath,

                    targetFile);

        } else if (ArchiveUtils.isZipFile(archive)) {

            decompressFileInZipArchive(archive, srcLogFilePath, targetFile);

        }

    }



    public static String getParentDir(String fullPath, String pathSep) {

        if ((fullPath ==  null) || (fullPath.length() == 0)) {

            return "";

        }

        if ((pathSep == null) || (pathSep.isEmpty())) {

            pathSep = File.separator;

        }

        int sepLoc = fullPath.lastIndexOf(pathSep);

        if (sepLoc == -1) {

            return "";

        } else {

            return fullPath.substring(0, sepLoc + 1);

        }

    }



    public static String getParentDir(String fullPath) {

        return getParentDir(fullPath, null);

    }



    public static String getFileName(String fullPath) {

        if ((fullPath ==  null) || (fullPath.length() == 0)) {

            return "";

        }

        return (new File(fullPath)).getName();

    }



    public static String getFileNameWithoutExt(String fullPath) {

        String fileName = getFileName(fullPath);

        return getSubStringByRegEx(fileName, "^([^\\.\\d]+)", 0);

    }



    /**

     * Find the nth (desiredMatchIndex) occurrence of a RegEx search string, and return the

     * specified capture group (sub-search-string) within that matched search string

     */

    private static String getSubStringByRegEx(String srcStr,

                                              String regEx,

                                              int regExGroup,

                                              int desiredMatchIndex) {

        String retStr = "";

        Pattern regexMatch = Pattern.compile(regEx);

        Matcher matcher = regexMatch.matcher(srcStr);

        int matchCount = 0;

        while (matcher.find()) {

            matchCount++;

            try {

                int startIdx = matcher.start(regExGroup);

                int endIdx = matcher.end(regExGroup);

                retStr = srcStr.substring(startIdx, endIdx);

            } catch (Exception e) {

                // If no match, return empty string

            }

            if (matchCount == desiredMatchIndex) {

                break;

            }

        }

        return retStr;

    }



    /**

     * Find the 1st occurrence of a RegEx search string, and return the specified capture group

     * (sub-search-string) of that search string.

     */

    public static String getSubStringByRegEx(String srcStr, String regEx, int regExGroup) {

        return getSubStringByRegEx(srcStr, regEx, regExGroup, 1);

    }



    /**

     * Find the last occurrence of a RegEx search string, and return the specified capture group

     * (sub-search-string) of that search string.

     */

    public static String getLastSubStringByRegEx(String srcStr, String regEx, int regExGroup) {

        return getSubStringByRegEx(srcStr, regEx, regExGroup, Integer.MAX_VALUE);

    }



    public static String getFileListStr(List<RAFile> raFiles) {

        StringBuffer retBuff = new StringBuffer(100);

        int totalSizeInBytes = 0;

        for (int i = 0; i < raFiles.size(); i++) {

            totalSizeInBytes += raFiles.get(i).getSizeInBytes();

            retBuff.append(i + ". " + raFiles.get(i).toDisplayString() + "\n");

        }

        retBuff.append("\nTotal Size = " + (totalSizeInBytes / 1024) + "." +

                padZeros(Integer.toString((totalSizeInBytes % 1024)), 3) + " KB");

        return retBuff.toString();

    }



    public static String formatElapsedTime(long timeInMillis) {

        if (timeInMillis == 0L) {

            return "0";

        }

        String sign = (timeInMillis < 0) ? "-" : "+";

        timeInMillis = Math.abs(timeInMillis);

        long days = timeInMillis / 86400000L;

        long dayFraction = timeInMillis % 86400000L;

        long hours = dayFraction / 3600000L;

        long hourFraction = dayFraction % 3600000L;

        long minutes = hourFraction / 60000L;

        long minuteFraction = hourFraction % 60000L;

        long seconds = minuteFraction / 1000L;

        long secondFraction = minuteFraction % 1000L;

        return sign +

               ((days > 0) ? days + "d " : "") +

               ((hours > 0) ? hours + "h " : "") +

               ((minutes > 0) ? minutes + "m " : "") +

               ((minuteFraction > 0) ? seconds + "." + secondFraction + "s" : "");

    }



    public static String formatElapsedTime(Date myDate) {

        return formatElapsedTime(myDate.getTime());

    }



    public static boolean arrayContains(Object[] objArr, Object value) {

        if ((objArr == null) || (objArr.length == 0) || (value == null)) {

            return false;

        }

        for (Object objArrValue : objArr) {

            if (objArrValue.equals(value)) {

                return true;

            }

        }

        return false;

    }



    /**

     * Trim the string, prepend all " " characters with "\"

     * @param inputStr  The input string with spaces in it

     * @return A string with spaces escaped

     */

    public static String escapeSpaces(String inputStr) {

        return inputStr.trim().replaceAll("\\s", Matcher.quoteReplacement("\\ "));

    }



}

