/***********************************************************
 * Copyright 2011 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core.fileops;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vide.vlogbrowser.core.DefaultLogBrowserManager;
import com.vmware.vide.vlogbrowser.core.ILogBrowserManager;
import com.vmware.vide.vlogbrowser.core.consts.LogBrowserConsts;
import com.vmware.vide.vlogbrowser.core.model.LogSystemType;
import com.vmware.vide.vlogbrowser.core.model.LogxFile;
import com.vmware.vide.vlogbrowser.core.parser.ConfigParser;
import com.vmware.vide.vlogbrowser.core.utils.Preferences;

public class LogxFileSetCreator {

    private List<LogxFile> myLxFiles;
    private String projectContainer = "";
    
    private static final Logger logger = LoggerFactory.getLogger(LogxFileSetCreator.class);

    public LogxFileSetCreator() {
        this.myLxFiles = new ArrayList<LogxFile>();
    }

    public List<LogxFile> getLxFiles() {
        return myLxFiles;
    }

    public void setProject(String name) {
        projectContainer = name;
    }

    public String getProject() {
        return projectContainer;
    }

    public String checkLogxFilenames() {
        for (LogxFile lxFile : myLxFiles) {
            if (! lxFile.isCreateThisFile()) {    // if this table row is not enabled, skip it
                continue;
            }
            String fileName = lxFile.getLogxFileName();
            if ((fileName == null) || (fileName.length() == 0)) {
                return "File name for '" + lxFile.toSummaryString() + "' must be specified";
            }
            if ((new File(fileName)).getParent() != null) {
                return "File name for '" + lxFile.toSummaryString() + "' must not include a" +
                    " parent directory.";
            }
            int dotLoc = fileName.lastIndexOf('.');
            if (dotLoc != -1) {
                String ext = fileName.substring(dotLoc + 1);
                if (ext.equalsIgnoreCase("logx") == false) {
                    return "File extension for '" + lxFile.toSummaryString() +
                        "' must be \"logx\"";
                }
                if (dotLoc == 0) {
                    return "File name for '" + lxFile.toSummaryString() + "' must be specified";
                }
            }
            if (! isUniqueInProject(lxFile.getLogxFileName())) {
                return "File name " + lxFile.getLogxFileName() + " already exists. It" +
                    " must be unique within project " + projectContainer;
            }
            if (! isFileUniqueInTable(lxFile)) {
                return "File name " + lxFile.getLogxFileName() + " is already in use in" +
                    " the table.";
            }
        }
        return null;
    }

    /**
     * Check to see if the proposed file name is unique in the project container
     * @param proposedName A String containing the proposed file name
     * @return true if unique, or false otherwise
     */
    private boolean isUniqueInProject(String proposedName) {
        String projectDir = Preferences.getInstance().getProperty(Preferences.ROOT_DIR);
        File fProjectDir = new File(projectDir);
        if (!fProjectDir.exists()) {
            return false;
        }
        
        for (File f: fProjectDir.listFiles()) {
            if (f.isDirectory()) {
                File file = new File(f, proposedName);
              if (file.exists()) {
                  return false;
              }
            }
        }
        return true;
    }

    /**
     * Check to see if the file name is unique in the table
     * @param lxFile
     * @return
     */
    private boolean isFileUniqueInTable(LogxFile lxFile) {
        for (LogxFile otherLxFile : myLxFiles) {  // Ensure LxFile is unique within the table
            if ( (otherLxFile.isCreateThisFile()) && (! lxFile.equals(otherLxFile)) &&
                 (lxFile.getLogxFileName().equals(otherLxFile.getLogxFileName())) ) {
                return false;
            }
        }
        return true;    // proposed name is unique
    }

    /**
     * Propose a (unique) name for a .logx file that will be created. This value is placed
     * in the ".logx name" column, to streamline the wizard flow.
     * @param srcHost
     * @param logFormat
     * @param isVMSuppArc
     * @return
     */
    public String getUniqueLogxFilename(String srcHost, String logFormat, boolean isVMSuppArc) {
        String lxFilenamePrefix = srcHost + "_" + logFormat + ((isVMSuppArc) ? "_vmsp" : "");
        lxFilenamePrefix = lxFilenamePrefix.replace(' ', '_');
        int count = 1;
        String uniqueNumStr = "";
        while (! isUniqueInProject(lxFilenamePrefix +
                                           uniqueNumStr +
                                           LogBrowserConsts.LOGX_FILE_EXTENSION)) {
            uniqueNumStr = Integer.toString(count++);
        }
        return lxFilenamePrefix + uniqueNumStr + LogBrowserConsts.LOGX_FILE_EXTENSION;
    }

    /**
     * Generate a name for a .logx file based on the srcHost+logFormat combination.
     *
     * @param srcHost
     * @param logFormat
     * @param isVMSuppArc
     * @return
     */
    public String getLogxFilename(String srcHost, String logFormat, boolean isVMSuppArc) {
        String lxFilenamePrefix = srcHost + "_" + logFormat + (isVMSuppArc ? "_vmsp" : "");
        lxFilenamePrefix = lxFilenamePrefix.replace(' ', '_');
        return lxFilenamePrefix + LogBrowserConsts.LOGX_FILE_EXTENSION;
    }

    /**
     * Based on the values input on the previous two wizard pages, create LogxFile objects
     * encapsulating that data and add them to the table. This is called from #setVisible()
     * @throws Exception
     */
    public void addLogxFilesToList(
                String hostname,
                LogxFile selectedFormat,
                String selectedFormatName,
                String tgzFilepath,
                String parentDir,
                String filePatterns
            ) throws Exception {

        // remember host used for next session
        ILogBrowserManager manager = new DefaultLogBrowserManager();
        manager.setRecentHost(hostname);
        String formatToSave = selectedFormat == null ? selectedFormatName :
            selectedFormat.getCategory() + "/" + selectedFormat.getLogFormatName();
        manager.setRecentHostFormat(hostname, formatToSave);

        // clear previous files
        myLxFiles.clear();

        boolean isStdPkg = selectedFormatName.equals(LogBrowserConsts.LOG_FORMAT_STD_PKG);
        boolean isVMSuppArc = selectedFormatName.equals(LogBrowserConsts.LOG_FORMAT_VMSUPPORT_ARCHIVE);
        if (isStdPkg || isVMSuppArc) {  // Already have default paths to the log files
            LogxFile[] templates = null;
            if (isStdPkg) {
                // use log formats for specified host
                templates = manager.getLogFormatsByHost(hostname, true);
            }
            String orgHostname = hostname;
            if (isVMSuppArc) {
                // find host version for TGZ file and use log formats for this host/version
                LogSystemType tgzSysType = manager.getTgzFileType(hostname, tgzFilepath);
                if (tgzSysType != null) {
                    // get matching templates
                    templates = manager.getLogFormatsByHosttype(tgzSysType, tgzFilepath);

                    // get hostname where support log was generated
                    orgHostname = tgzSysType.getDir();
                }
            }

            if (templates != null) {
                for (LogxFile template : templates) {
                    String logFormatName = template.getLogFormatName();
                    String category = template.getCategory();
                    URL templateFile = template.getTemplateURL();
                    LogSystemType systemType = template.getSystemType();
                    LogxFile lxFile = new LogxFile(logFormatName, category, templateFile, systemType);
                    lxFile.setSrcHost(hostname);
                    lxFile.setLogxFileName(getUniqueLogxFilename(orgHostname, logFormatName, isVMSuppArc));
                    lxFile.setCreateThisFile(true);
                    lxFile.setOpenEditor(false);
                    if (isVMSuppArc) {
                        lxFile.setArchivePath(parentDir + filePatterns);
                    }
                    if (! myLxFiles.contains(lxFile)) {
                        myLxFiles.add(lxFile);
                    }
                }
            }
        } else {
            // Not part of a log set, so we needed (and got) more specific info
            // create new SystemType with a different file patterns
            LogxFile selectedTemplate = selectedFormat;
            String filePtrns = "";
            if (filePatterns.startsWith(LogBrowserConsts.REGULAR_EXP_MSG_PREAMBLE)) {
                filePtrns = selectedTemplate.getSystemType().getFileRegex();
            } else {
                filePtrns = filePatterns;
            }
            LogSystemType sysType = selectedTemplate.getSystemType();
            LogSystemType newSysType = new LogSystemType(sysType.getSystem(), sysType.getVersion(),
                    parentDir, filePtrns, sysType.getLiveLogFilename(), sysType.isFileNumDescending(), sysType.getSortMethod());

            // create new log file
            LogxFile lxFile = new LogxFile(selectedTemplate.getLogFormatName(),
                selectedTemplate.getCategory(), selectedTemplate.getTemplateURL(), newSysType);
            lxFile.setSrcHost(hostname);
            String logxFilename = getUniqueLogxFilename(hostname, selectedTemplate.getLogFormatName(), isVMSuppArc);
            lxFile.setLogxFileName(logxFilename);
            lxFile.setCreateThisFile(true);
            lxFile.setOpenEditor(true);
            if (! filePatterns.startsWith(LogBrowserConsts.REGULAR_EXP_MSG_PREAMBLE)) {
                String[] logFileArr = filePtrns.split(";");
                List<String> logFileList = new ArrayList<String>();
                for (String logFileStr : logFileArr) {
                    logFileList.add(logFileStr + "$");
                }
                lxFile.setFileRegexes(logFileList);
            }
            if (! myLxFiles.contains(lxFile)) {     // uses overridden LogxFile#equals() method
                myLxFiles.add(lxFile);
            }
        }
    }

    public static File getLogxResource(String containerName) throws Exception {
//        IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
//        IProject project = null;
//        IResource resource = root.findMember(new Path(containerName));
//        if ((resource == null) || (!resource.exists() || !(resource instanceof IContainer))) {
//            project = root.getProject(containerName);
//            String descStr = containerName.replaceAll("[\\:/\\\\]", "_");  // replace any file separators
//            IProjectDescription projDesc =
//                ResourcesPlugin.getWorkspace().newProjectDescription(descStr);
//            projDesc.setLocation(null);   // null means use default location
//            project.create(projDesc, null);
//            project.open(null);
//            resource = project;
//        }
//        if (!resource.exists() || !(resource instanceof IContainer)) {
//            return null;
//        }
//        return resource;
        String projectDir = Preferences.getInstance().getProperty(Preferences.ROOT_DIR);
        File fProjectDir = new File(projectDir);
        if (!fProjectDir.exists()) {
            return null;
        }
        
        for (File f: fProjectDir.listFiles()) {
            if (f.isDirectory()) {
                if (f.getName().equals(containerName)) {
                    return f;
                }
            }
        }
        return null;
    }

    public static boolean createLogxFile(String containerName, LogxFile lxFile) throws Exception {
        String archivePath = lxFile.getArchivePath();
        String logFilePaths = lxFile.getSystemType().getFileRegex();
        String srcHostChoice = lxFile.getSrcHost();
        String parentDir = lxFile.getSystemType().getDir();
        List<String> fileRegexList = lxFile.getFileRegexes();
        String tmpFileName = lxFile.getLogxFileName();
        if (!tmpFileName.endsWith(".logx")) {
            tmpFileName += ".logx";
        }
        String fileName = tmpFileName;

        // create a sample file
        logger.debug("Creating " + fileName);
//        IResource resource = getLogxResource(containerName);
//        File resource = getLogxResource(containerName);
//        IContainer container = (IContainer) resource;
        File container = getLogxResource(containerName);
        if (container != null) {
        final File file = new File(container, fileName);
        try {
            // Get initial inputStream with XML config that is missing log file paths
//            File templatePath = new File(lxFile.getTemplateFilePath());
//            InputStream beforeStream = new FileInputStream(templatePath);
            InputStream beforeStream = lxFile.getTemplateURL().openStream();
            ConfigParser cfgParser = null;
//            try {
                cfgParser = new ConfigParser(null, beforeStream);
//            } catch (Exception e) {
//                throw new CoreException(null);
//            }
            beforeStream.close();

            // Set the source log file elements in the XML config (will become a .logx file)
            // based on the LogxFile objects backing the table on the 3rd wizard screen.
            String[] fileRegexArr = null;
            if ((fileRegexList != null) && (fileRegexList.size() > 0)) {
                fileRegexArr = fileRegexList.toArray(new String[fileRegexList.size()]);
            } else {
                fileRegexArr = new String[]{logFilePaths};
            }
            cfgParser.addSrcLogFileNode(srcHostChoice, parentDir, archivePath, fileRegexArr);

//            String xmlString = cfgParser.getXmlFromDocument();
//            byte[] xmlByteArr = xmlString.getBytes();
            // The updated inputStream including the log file path element
//            InputStream afterStream = new ByteArrayInputStream(xmlByteArr);

//            if (file.exists()) {
//                file.setContents(afterStream, true, true, null);
//            } else {
//                file.create(afterStream, true, null);
//            }
//            afterStream.close();
            if (!file.exists()) {
                file.createNewFile();
            }
            FileWriter writer = new FileWriter(file);
            writer.write(cfgParser.getXmlFromDocument());
            writer.close();
        } catch (IOException e) {
            logger.error(e.getMessage(), e);
        }
        }
        return true;
    }

}
