/***********************************************************

 * Copyright 2010 VMware, Inc.  All rights reserved.

 * -- VMware Confidential

 ***********************************************************/



package com.vmware.vide.vlogbrowser.core.fileops;



import java.io.File;



import com.vmware.vide.vlogbrowser.core.parser.RAFile;

import com.vmware.vide.vlogbrowser.core.utils.Utils;



/**

 * Objects of this class represent a temporary destination directory for VM Support Logs (i.e.

 * .tgz files). The LogFileManager.stageLogFiles() method uses this class to abstract away some

 * of the complexity of handling a file decompress destination. This temp directory will be a

 * subdirectory of the target system's temp directory. The temp directory uses a hash of several

 * values to be unique and is dedicated to hosting the set of log files described in a specific

 * /root/SourceLogFiles/Source/ID XML element in a specific .logx configuration file. In other

 * words, if needed for .tgz extraction, there is one temporary unpack directory for each

 * /root/SourceLogFiles/Source/ID element in each .logx file.

 */

public class TempUnpackDir {



    public static final String UNIX_TEMP_DIR_PARENT = "/tmp";



    private String cfgFileId;   // the <logx_ID> element -- (unique ID for the .logx file)



    // Next three strings are ':' delimited in the .logx file in element /root/SourceLogFiles/Source/ID

    // They identify a unique set of File names (possibly via RegExes that expand to file names) that

    // are found at the location named at /root/SourceLogFiles/Source/ID

    private String srcFileHost;

    private String srcArchivePath;

    private String srcFileParentDir;



    // To be unique, the unpack directory will be a full path that contains a hash of the above four values

    private String directory;



    public TempUnpackDir(String cfgFileId, RAFile raFile) throws Exception {

        this.cfgFileId = cfgFileId;



        this.srcFileHost = raFile.getSrcFileHost();

        this.srcArchivePath = raFile.getSrcArchivePath();

        this.srcFileParentDir = LogFileUtils.getParentDir(raFile.getSrcFilePath());



        // Local or Remote system will be Unix-variant, since Windows systems will use the Apache TAR library

        // temp path format is /tmp/vLogBrowser-nnn, which will be unique for different users and cross-permissions

        String tempDirParent = UNIX_TEMP_DIR_PARENT + "/vLogBrowser-";



        this.directory = tempDirParent +

            LogFileUtils.computeMD5Hash(cfgFileId + srcFileHost + srcArchivePath + srcFileParentDir);

    }



    public void createLocal() throws Exception {

        File myDir = new File(directory);

        if ((! new File(directory).mkdirs()) && ((! myDir.exists()) || (! myDir.canWrite()))) {

            throw new Exception("Temporary unpack directory " + directory + " for " +

                    srcArchivePath + " could not be created or re-used.");

        }

    }



    public String getRemoteCreateCmd() {

        return "mkdir -p " + toString();

    }



    public String getSourceFileHost() {

        return srcFileHost;

    }



    @Override

    public String toString() {

        return directory + Utils.getPathSep(srcFileHost);

    }



    @Override

    public boolean equals(Object otherObj) {

        if (otherObj instanceof TempUnpackDir) {

            TempUnpackDir otherDir = (TempUnpackDir) otherObj;

            if ( (otherDir.cfgFileId.equals(cfgFileId)) &&

                 (otherDir.srcFileHost.equals(srcFileHost)) &&

                 (otherDir.srcArchivePath.equals(srcArchivePath)) &&

                 (otherDir.srcFileParentDir.equals(srcFileParentDir)) ) {

                return true;

            }

        }

        return false;

    }



    @Override

    public int hashCode() {

        return directory.hashCode();

    }



}

