/***********************************************************
 * Copyright 2010 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core.model;

import java.util.regex.Pattern;

import com.vmware.vide.vlogbrowser.core.consts.LogBrowserConsts.BooleanConnector;

/**
 * Objects of the class FilterItem are stored in a 2D list in a LogFilter object, which backs
 * the FilterDialog GUI and serves as the filter engine for the LogTableViewer.
 * Each filterItem corresponds to a single filter term displayed in the FilterDialog:
 * Boolean Connector AND/OR/<blank>(skip) | Negation Flag "NOT" | Keyword | Keyword is RegEx flag
 */
public class FilterItem {
    private BooleanConnector boolConn;
    private boolean negateFlag;
    private String keyword;
    private boolean regexFlag;
    private Pattern regexPattern;

    public FilterItem() {
        init();
    }

    public void init() {
        this.boolConn = BooleanConnector.SKIP;
        this.negateFlag = false;
        this.keyword = "";
        this.regexFlag = false;
        this.regexPattern = null;
    }

    public BooleanConnector getBoolConnector() {
        return boolConn;
    }

    public void setBoolConnector(BooleanConnector boolConn) {
        this.boolConn = boolConn;
    }

    public boolean isNegateFlagSet() {
        return negateFlag;
    }

    public void setNegateFlag(boolean negateFlag) {
        this.negateFlag = negateFlag;
    }

    public String getKeyword() {
        return keyword;
    }

    public void setKeyword(String keyword) {
        this.keyword = keyword;
        if (regexFlag) {
            regexPattern = Pattern.compile(keyword);
        }
    }

    public boolean isRegexFlagSet() {
        return regexFlag;
    }

    public void setRegexFlag(boolean regexFlag) {
        this.regexFlag = regexFlag;
        if (regexFlag) {
            regexPattern = Pattern.compile(keyword);
        }
    }

    /**
     * Evaluate this filter item (filter term) against the source string (log entry).
     */
    public boolean evaluate(String sourceStr) {
        boolean found = (regexFlag) ? regexPattern.matcher(sourceStr).find()
                                    : (sourceStr.indexOf(keyword) > -1);
        return (negateFlag) ? (! found) : found;
    }

    public void setBoolConnector(String boolConnStr) {
        if (BooleanConnector.AND.toString().equals(boolConnStr)) {
            boolConn = BooleanConnector.AND;
        } else if (BooleanConnector.OR.toString().equals(boolConnStr)) {
            boolConn = BooleanConnector.OR;
        } else {
            boolConn = BooleanConnector.SKIP;
        }
    }

}
