/***********************************************************

 * Copyright 2010 VMware, Inc.  All rights reserved.

 * -- VMware Confidential

 ***********************************************************/



package com.vmware.vide.vlogbrowser.core.model;



import java.util.Comparator;

import java.util.Vector;

import java.util.concurrent.Future;



import com.vmware.vide.vlogbrowser.core.consts.LogBrowserConsts;

import com.vmware.vide.vlogbrowser.core.fileops.LogItemListAddTask;

import com.vmware.vide.vlogbrowser.core.parser.MultiMatcher;

import com.vmware.vide.vlogbrowser.core.utils.TaskScheduler;



/**

 * Class used by Log Browser to store loaded log entries.

 * This class has 2 lists: all items list and filtered items list. API similar to Vector class

 * return items from filtered list. Method getUnfilteredItems() returns all log entries in the

 * current the sorted order.

 */

public class FilterList {

    Vector<ListEventListener> fListeners;

    ILogItemList fAllItems;

    ILogItemList fFilteredItems;

    MultiMatcher fMatcher;

    private static final String FILTERED_SUFIX = ".filtered";

    private static final String UNFILTERED_SUFIX = ".unfiltered";



    /** Creates new list (based on Vector) */

    public FilterList(String resName) {

        super();

        LogItemListFactory itemFactory = LogItemListFactory.getInstance();

        String resNameBase = resName.replaceFirst(LogBrowserConsts.LOGX_FILE_EXTENSION, "");

        fAllItems = itemFactory.createLogItemList(itemFactory.getDefaultType(), resNameBase+UNFILTERED_SUFIX);

        fFilteredItems = itemFactory.createLogItemList(itemFactory.getDefaultType(), resNameBase+FILTERED_SUFIX);

    }



    /** Add listener for list change events */

    public void addListEventListener(ListEventListener listener) {

        if (fListeners == null)

            fListeners = new Vector<ListEventListener>();

        fListeners.add(listener);

    }



    /** Remove listener for list change events */

    public void removeListEventListener(ListEventListener listener) {

        if (fListeners != null)

            fListeners.remove(listener);

    }



    /** Return all list change event listeners */

    public ListEventListener[] getListeners() {

        return fListeners.toArray(new ListEventListener[fListeners.size()]);

    }



    /** Notify all listeners when list contents changes */

    private void notifyListeners() {

        if (fListeners == null)

            return;



        for (int i = 0; i < fListeners.size(); i++) {

            ListEventListener listener = fListeners.get(i);

            listener.listChanged();

        }

    }



    

    /** Adds element to both list: allItems and filteredItems */

    public boolean add(LogItem logItem) {

//        boolean added = fAllItems.add(logItem);

//        fFilteredItems.add(logItem);

//        notifyListeners();

//        return added;

//        class Task implements Runnable {

//            ILogItemList list;

//            LogItem logItem;

//            public Task(ILogItemList list, LogItem logItem) {

//                this.list = list;

//                this.logItem = logItem;

//            }

//            @Override

//            public void run() {

//               list.add(logItem);

//            }

//        };

//        System.out.println(Thread.currentThread().getName());

        fAllItems.add(logItem);

        fFilteredItems.add(logItem);

//        Future<?> future = TaskScheduler.add(new LogItemListAddTask(fAllItems, logItem));

//        Future<?> future2 = TaskScheduler.add(new LogItemListAddTask(fFilteredItems, logItem));

//        try {

//            future.get();

////            future2.get();

//        } catch (InterruptedException e) {

//            // TODO Auto-generated catch block

//            e.printStackTrace();

//        } catch (Exception e) {

//           return false;

//        }

        return true;

    }

    

    /** Adds elements to both list: allItems and filteredItems */

    public boolean addAll(ILogItemList logItemList) {

        boolean added = fAllItems.addAll(logItemList);

        fFilteredItems.addAll(logItemList);

        notifyListeners();

        return added;

    }



    /** Adds element to all items list */

    public boolean addIntoAllItems(LogItem logItem) {

        boolean added = fAllItems.add(logItem);

        return added;

    }

    

    /** Adds elements to this filtered list */

    public boolean populateFilteredListWithAllItems() {

        boolean added = fFilteredItems.addAll(fAllItems);

//        notifyListeners();

        return added;

    }

    

    public ILogItemList getAllItems() {

        return fAllItems;

    }



    /**

     * Set the Matcher which specifies which elements shall be filtered.

     * This will remove the current Matcher or MatcherEditor and re-filter the entire list.

     */

    public void setMatcher(MultiMatcher m) {

        fFilteredItems.clear();



        fMatcher = m;



        // check if removing filter

        if (m == null) {

            fFilteredItems.addAll(fAllItems);

            return;

        }



        for (int i = 0; i < fAllItems.size(); i++) {

            LogItem logItem = fAllItems.get(i);

            if (m.matches(logItem)) {

                fFilteredItems.add(logItem);

            }

        }

    }



    /** Returns the number of filtered items in this list */

    public int size() {

        return fFilteredItems.size();

    }



    /** Returns the index of the first occurrence of the specified element */

    public int indexOf(LogItem logItem) {

        return fFilteredItems.indexOf(logItem);

    }



    /** Removes all of the elements from this list */

    public void clear() {

        fFilteredItems.clear();

        fAllItems.clear();

    }



    /** Returns filtered element at the specified index in this list */

    public LogItem get(int index) {

        return fFilteredItems.get(index);

    }



    public ILogItemList getUnfilteredItems() {

        return fAllItems;

    }



    public ILogItemList getFilteredItems() {

        return fFilteredItems;

    }



    public void setComparator(Comparator<LogItem> comparator) {

        fAllItems.sort(comparator);



        // if filtered, then re-apply filter

        if (fMatcher != null) {

            setMatcher(fMatcher);

        } else {

            fFilteredItems.clear();

            fFilteredItems.addAll(fAllItems);

        }

    }



    public void updateLogItem(long i, LogItem item) {

        fAllItems.updateLogItem(i, item);

        fFilteredItems.updateLogItem(i, item);

    }

}

