/***********************************************************

 * Copyright 2009 VMware, Inc.  All rights reserved.

 * -- VMware Confidential

 ***********************************************************/



package com.vmware.vide.vlogbrowser.core.model;



import java.text.DateFormat;

import java.text.SimpleDateFormat;

import java.util.Date;



import com.vmware.vide.vlogbrowser.core.parser.LogDateFormat;



/**

 * The LogDate class extends a Date (stored as a long -- milliseconds since 1/1/1970) and is

 * used by a LogItem object--a parsed log line--to store the timestamp of that log line.

 * LogDate class extends Date, and behaves similarly to a Date, with one major difference:

 * It has a reference to the LogDateFormat object which created it. LogDateFormat helps LogDate

 * compute the correct timestamp to return to some caller to overcome the missing year problem.

 * Many log formats, including vmkernel, are missing the year. To finish timestamps that are

 * missing the year, LogDateFormat creates LogDate objects, one for each log line,

 * using the default year: 1970. Later, when a caller requests a particular LogDate object's

 * timestamp value, it calls the LogDateFormat object that created it to compute a correct

 * timestamp with appropriate year. This is necessary because at the time of instantiation,

 * the LogDate object's proper year is not known. Only after the entire log has been parsed

 * can the year be inferred by the LogDateFormat object using the timestamp on the log file

 * and subtracting the number of year rollovers that were detected in the log file

 * (e.g. December rolls to January) Please see the LogDateFormat class for more information.

 */

public class LogDate extends Date {



    private static final long serialVersionUID = 1L;

    private LogDateFormat logDateFormat;

    private int numRollovers;

    public static DateFormat outputDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS Z");



    public LogDate(Long dateInMillis, LogDateFormat logDateFormat, int numRollovers) {

        super(dateInMillis);

        this.logDateFormat = logDateFormat;

        this.numRollovers = numRollovers;

    }



    @Override

    public long getTime() {

        long time = super.getTime();

        if (time == 0)

            return time;



        return logDateFormat.correctYearlessDate(time, numRollovers);

    }



    @Override

    public void setTime(long time) {

        System.out.println("Warning: unimplemented setTime() called on a LogDate object" +

                " (which extends Date)");

    }



    @Override

    public boolean after(Date when) {

        return (this.compareTo(when) > 0);

    }



    @Override

    public boolean before(Date when) {

        return (this.compareTo(when) < 0);

    }



    @Override

    public int compareTo(Date anotherDate) {

        long thisDate = this.getTime();

        long otherDate = anotherDate.getTime();



        if (thisDate < otherDate) {

            return -1;

        } else if (otherDate == thisDate) {

            return 0;

        } else {

            return 1;

        }

    }



    @Override

    public boolean equals(Object obj) {

        if ((obj instanceof LogDate) && (this.compareTo((LogDate)obj) == 0)) {

                return true;

        } else {

            return false;

        }

    }



    @Override

    public String toString() {

        if (super.getTime() > 0L) {

            long timestamp = getTime();

            timestamp += logDateFormat.getDateDelta();

            return outputDateFormat.format(timestamp);

        } else {    // 0 (1/1/1970, 12 midnight) is a special case -- display an empty string

            return "";

        }

    }



    public LogDateFormat getLogDateFormat() {

        return logDateFormat;

    }



    public void setLogDateFormat(LogDateFormat logDateFormat) {

        this.logDateFormat = logDateFormat;

    }



}

