/***********************************************************
 * Copyright 2010-2011 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core.model;

import java.util.ArrayList;
import java.util.List;

import com.vmware.vide.vlogbrowser.core.consts.LogBrowserConsts;
import com.vmware.vide.vlogbrowser.core.consts.LogBrowserConsts.BooleanConnector;

/**
 * Objects of the LogFilter class store a filter that can be applied to the log. The filter
 * can be simple, with a single plain-text keyword match, or it can match up to four terms
 * (which may be regular expressions) using AND/OR/NOT operators. A logFilter object contains
 * a 2D list of FilterItem objects, and serves as the model that backs the FilterDialog.
 * After the logFilter object has been configured by the user (using the FilterDialog), the
 * logFilter is applied to the table of log entries and serves as the filtering engine.
 */

public class LogFilter {

    private String name;

    // This 2D List allows dynamic addition of new rows/items
    private List<List<FilterItem>> filterItemMatrix;
    private int numSurroundingLines;
    private boolean filterEnabled;
    private boolean isCaseSensitive;

    public static final int FIRST_ROW_IDX = 0;
    public static final int FIRST_COLUMN_IDX = 0;

    private int maxFilterItemIndex;

    public LogFilter(String name, int numSurroundingLines, int maxFilterColumns) {
        setName(name);
        filterItemMatrix  = new ArrayList<List<FilterItem>>();
        setNumSurroundingLines(numSurroundingLines);
        filterEnabled = false;
        setMaxFilterColumns(maxFilterColumns);
    }

    public String getName() {
        return name;
    }

    public void setName(String newName) {
        name = (newName == null) ? "" : newName;
    }

    public int getNumSurroundingLines() {
        return numSurroundingLines;
    }

    public void setNumSurroundingLines(int surroundingLines) {
        if ((surroundingLines < 0) || (surroundingLines > LogBrowserConsts.MAX_NUM_ADD_LINES)) {
            surroundingLines = 0;
        }
        numSurroundingLines = surroundingLines;
    }

    public int getMaxFilterColumns() {
        return maxFilterItemIndex + 1;
    }

    public void setMaxFilterColumns(int maxFilterColumns) {
        maxFilterItemIndex = maxFilterColumns - 1;
    }

    public void setFilterEnabled(boolean filterEnabled) {
        this.filterEnabled = filterEnabled;
    }

    public boolean isFilterEnabled() {
        return filterEnabled;
    }

    /**
     * If necessary, expand the 2D list of Filter Items to a sufficient capacity to store a
     * filter item in the desired position.
     * This method is called before any get/set filter item operations.
     */
    private void checkIndexes(int rowIndex, int filterItemIndex) throws Exception {
        if (rowIndex >= filterItemMatrix.size()) {   // User can add any number of rows in Filter Dlg
            for (int i = filterItemMatrix.size(); i <= rowIndex; i++) {
                filterItemMatrix.add(new ArrayList<FilterItem>());
            }
        } else if (rowIndex < 0) {
            throw new Exception("Problem getting/setting filter item: row index " + rowIndex +
                    " is less than 0");
        }

        List<FilterItem> myFilterRow = filterItemMatrix.get(rowIndex);
        if ((filterItemIndex >= myFilterRow.size()) && (filterItemIndex <= maxFilterItemIndex)) {
            for (int i = myFilterRow.size(); i <= filterItemIndex; i++) {
                myFilterRow.add(new FilterItem());
            }
        } else if ((filterItemIndex < 0) || (filterItemIndex > maxFilterItemIndex)) {
            throw new Exception("Problem getting/setting filter item: filter item index " +
                    filterItemIndex + " out of the range 0 - " + maxFilterItemIndex);
        }
    }

    public FilterItem getFilterItem(int rowIndex, int filterItemIndex) throws Exception {
        checkIndexes(rowIndex, filterItemIndex);
        return filterItemMatrix.get(rowIndex).get(filterItemIndex);
    }

    public void setFilterItem(int rowIndex,
                              int filterItemIndex,
                              FilterItem filterItem) throws Exception {
        checkIndexes(rowIndex, filterItemIndex);
        filterItemMatrix.get(rowIndex).set(filterItemIndex, filterItem);
    }

    public int getFilterRowCount() {
        return filterItemMatrix.size();
    }

    public int getFilterColumnCount(int row) {
        if (row < 0 || row >= filterItemMatrix.size()) {
            return 0;
        }
        List<FilterItem> filterRow = filterItemMatrix.get(row);
        return filterRow.size();
    }

    /*
     * Loop through the Filter Row, visiting each filter item one by one. When a term is by
     * itself, i.e., bounded by OR operator(s), check to see if the term is TRUE. If so,
     * return TRUE (and be done: shortcut evaluation). If false, continue evaluating terms.
     * For each string of terms encountered that need to be joined together with AND operators,
     * do that before considering the OR operators (which are of a lower precedence than AND).
     */
    private boolean evaluateFilterRow(List<FilterItem> myFilterRow, String sourceStr) {
        boolean matched = false;
        int columnIdx = FIRST_COLUMN_IDX;
        int columnCount = myFilterRow.size();
        while ((! matched) && (columnIdx < columnCount)) {
            if ( (columnIdx != FIRST_COLUMN_IDX) &&
                 (myFilterRow.get(columnIdx).getBoolConnector() == BooleanConnector.SKIP)) {
                // If a term has been "deactivated" by setting its Boolean Connector to SKIP, ignore it
                columnIdx++;
                continue;
            }
            matched = myFilterRow.get(columnIdx++).evaluate(sourceStr);

            while ( (columnIdx < columnCount) &&   // Consume any subsequent terms connected with 'AND'
                    (myFilterRow.get(columnIdx).getBoolConnector() != BooleanConnector.OR) ) {

                if (myFilterRow.get(columnIdx).getBoolConnector() == BooleanConnector.SKIP) {
                    columnIdx++;
                    continue;
                }

                matched = matched && myFilterRow.get(columnIdx).evaluate(sourceStr);
                columnIdx++;
            }
        }
        return matched;
    }

    /*
     * Loop through the Filter Item Matrix row by row, in much the same way as looping through
     * each Filter Row. Each filter row is evaluated as a unit, as the parentheses in the GUI
     * are meant to imply.
     */
    public boolean evaluateFilterMatrix(String sourceStr) {
        boolean matched = false;
        int rowIdx = FIRST_ROW_IDX;
        int rowCount = filterItemMatrix.size();
        int columnCount = (rowCount > rowIdx) ? filterItemMatrix.get(rowIdx).size() : -1;
        while ((! matched) && (rowIdx < rowCount)) {
            if ( (columnCount == 0) ||
                   ((rowIdx != FIRST_ROW_IDX) &&
                    (filterItemMatrix.get(rowIdx).get(FIRST_COLUMN_IDX)
                            .getBoolConnector() == BooleanConnector.SKIP))) {
                rowIdx++;
                columnCount = (rowCount > rowIdx) ? filterItemMatrix.get(rowIdx).size() : -1;
                continue;
            }
            matched = evaluateFilterRow(filterItemMatrix.get(rowIdx++), sourceStr);
            columnCount = (rowCount > rowIdx) ? filterItemMatrix.get(rowIdx).size() : -1;
            while ( (rowIdx < rowCount) &&
                        ((columnCount == 0) ||
                         (filterItemMatrix.get(rowIdx).get(FIRST_COLUMN_IDX)
                                 .getBoolConnector() != BooleanConnector.OR)) ) {

                if ( (columnCount == 0) ||
                     (filterItemMatrix.get(rowIdx).get(FIRST_COLUMN_IDX)
                             .getBoolConnector() == BooleanConnector.SKIP) ) {
                     rowIdx++;
                     columnCount = (rowCount > rowIdx) ? filterItemMatrix.get(rowIdx).size() : -1;
                     continue;
                 }

                matched = matched && evaluateFilterRow(filterItemMatrix.get(rowIdx), sourceStr);
                rowIdx++;
                columnCount = (rowCount > rowIdx) ? filterItemMatrix.get(rowIdx).size() : -1;
            }
        }
        return matched;
    }

    public boolean isJustAKeyword() {
        if (! name.isEmpty()) {
            return false;
        }
        if ((filterItemMatrix.size() > 0) && (filterItemMatrix.get(0).size() > 0)) {
            FilterItem firstFilterItem = filterItemMatrix.get(0).get(0);
            if ((firstFilterItem.isNegateFlagSet()) || (firstFilterItem.isRegexFlagSet())) {
                return false;
            }
        }
        for (List<FilterItem> filterItemRow : filterItemMatrix) {
            for (FilterItem filterItem : filterItemRow) {
                if (filterItem.getBoolConnector() != BooleanConnector.SKIP) {
                    return false;
                }
            }
        }
        return true;
    }

    public String getFirstKeyword() throws Exception {
        checkIndexes(0, 0);
        return filterItemMatrix.get(0).get(0).getKeyword();
    }

    public void setFirstKeyword(String keyword) throws Exception {
        checkIndexes(0, 0);
        filterItemMatrix.get(0).get(0).setKeyword(keyword);
    }

    public int getNumRows() {
        return filterItemMatrix.size();
    }

    public int getNumColumns(int rowNum) {
        if ((rowNum < filterItemMatrix.size()) && (filterItemMatrix.get(rowNum) != null)) {
            return filterItemMatrix.get(rowNum).size();
        }
        return 0;
    }

    public void printLogFilter() {
        System.out.println("***********************************");
        System.out.println("Log Filter name: " + name);
        System.out.println("Log Filter is enabled: " + filterEnabled);
        for (List<FilterItem> myFilterItemList : filterItemMatrix) {
            for (FilterItem filterItem : myFilterItemList) {
                System.out.print(filterItem.getBoolConnector().toString() +
                        ((filterItem.isNegateFlagSet()) ? " NOT " : " ") +
                        filterItem.getKeyword() +
                        ((filterItem.isRegexFlagSet()) ? " (RegEx)  " : "  "));
            }
            System.out.println("");
        }
        System.out.println("Log Filter surrounding lines: " + numSurroundingLines);
        System.out.println("***********************************");
    }

    public void setCaseSensitive(boolean isCaseSensitive) {
        this.isCaseSensitive = isCaseSensitive;
    }

    public boolean isCaseSensitive() {
        return isCaseSensitive;
    }

}
