/***********************************************************
 * Copyright 2009 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core.model;

import java.io.IOException;
import java.util.Date;

import com.vmware.vide.vlogbrowser.core.parser.LogFormat;
import com.vmware.vide.vlogbrowser.core.parser.RAFile;

/**
 * The LogItem class represents a line in a log file, parsed into fields (cells in a table).
 * It stores the index number to sequentially number a log entry and a Date object (a timestamp),
 * along with the offset and length of the log entry in the original log file. A caller requesting a
 * particular field (string) in a log entry (a field that is not the Date or sequence number,
 * which are stored in the LogItem) must supply the LogFormat object that originally created
 * the LogItem object, and the LogItem object will pass its own stored offset and length to
 * that LogFormat object to allow the desired field to be parsed from the source file and
 * returned to the caller. In addition, the logExtra string may also be returned, if available,
 * which is comprised of extra text that was found on subsequent lines below the log entry, but
 * that are still part of that log entry.
 */
public class LogItem {

    private boolean alwaysMatched;
    private boolean properlyFormed;
    private int groupId;       // used when showing group of entries

    public static final int EXTRA_LIMIT = 25000;

    // The following fields must be stored in this object
    private final long fileOffset;
    private int readLength;

    // This field is needed to reconstitute the date; the numYearRollovers cannot
    // be determined by random access, and the date might as well be stored,
    // since it is parsed
    private Date date;

    // The logIndexNum must be stored, since it cannot be parsed (it is a count of LogItems)
    private long logIndexNum;

    private final int logFileNum;

    public LogItem(boolean properlyFormed,
                   long fileOffset,
                   int readLength,
                   Date date,
                   long logIndexNum,
                   int logFileNum) {
        this.properlyFormed = properlyFormed;
        this.fileOffset = fileOffset;
        this.readLength = readLength;
        this.date = date;
        this.logIndexNum = logIndexNum;
        this.logFileNum = logFileNum;

        this.alwaysMatched = false;     // Can be set by filter if user picks "surrounding lines"
        this.groupId = 0;
    }

    public Object getLogField(int columnIndex, LogFormat myLogFormat) throws IOException {
        if (columnIndex == getLogIndex(myLogFormat)) {
            return new Long(logIndexNum);
        } else if (columnIndex == getDateIndex(myLogFormat)) {
            if (date != null) {
                return date;
            } else {
                return "";
            }
        } else {
            byte[] byteBuffer = new byte[readLength];
            myLogFormat.getRandAccFile(logFileNum).readBytes(byteBuffer, fileOffset);
            String logLine = new String(byteBuffer);
            if (properlyFormed) {
                return myLogFormat.getLogCell(columnIndex, logLine);
            } else {
                if (columnIndex == getDefaultIndex(myLogFormat)) {
                    return myLogFormat.getLogPreamble(logLine);
                } else {
                    return "";
                }
            }
        }
    }

    @SuppressWarnings("unchecked")
    public Comparable<Object> getLogFieldComparable(int columnIndex,
                                            LogFormat myLogFormat) throws IOException {
        return (Comparable<Object>)getLogField(columnIndex, myLogFormat);
    }

    public Object[] getLogFieldsArr(LogFormat myLogFormat) throws IOException {
        Object[] logFields = new Object[myLogFormat.getColumnNames().length];
        for (int i = 0; i < logFields.length; i++) {
            logFields[i] = getLogField(i, myLogFormat);
        }
        return logFields;
    }

    public String getLogExtra(LogFormat myLogFormat) throws IOException {
        byte[] byteBuffer = new byte[readLength];
        myLogFormat.getRandAccFile(logFileNum).readBytes(byteBuffer, fileOffset);
        return myLogFormat.getLogExtra(new String(byteBuffer, myLogFormat.getCharEncode()));
    }

    /**
     * Get all the raw bytes that belong to this log item (beginning at the offset, continuing
     * for the length). This is faster if you don't need to get the log entry fields individually.
     */
    public byte[] getLogEntryBytes(LogFormat myLogFormat) throws IOException {
        byte[] byteBuffer = new byte[readLength];
        myLogFormat.getRandAccFile(logFileNum).readBytes(byteBuffer, fileOffset);
        return byteBuffer;
    }

    public int getDefaultIndex(LogFormat myLogFormat) {
        return myLogFormat.getDefaultColumnIndex();
    }

    public String getDefaultColumnString(LogFormat myLogFormat) throws IOException {
        return getLogField(getDefaultIndex(myLogFormat), myLogFormat).toString();
    }

    public String toVerboseString(LogFormat myLogFormat) throws IOException {
        String retStr = "";
        retStr += "******************\n";
        for (int i = 0; i < myLogFormat.getColumnNames().length; i++) {
            retStr += i + ": '" + getLogField(i, myLogFormat).toString() + "'\n";
        }
        retStr += "Log Extra:\n" + getLogExtra(myLogFormat) + "\n";
        retStr += "******************\n";

        return retStr;
    }

    public String toBriefString(LogFormat myLogFormat) throws IOException {
        String retStr = "";
        for (int i = 0; i < myLogFormat.getColumnNames().length; i++) {
            String s = getLogField(i, myLogFormat).toString();

            // trim \0 chars (affecting scrolling), which can be in the first line of log
            if (s.length() > 0 && s.charAt(0) == '\0')
                s = s.replaceAll("\0", "");

            retStr += " | " + s;
        }
        String logExtra = getLogExtra(myLogFormat);
        if (logExtra.length() > 0) {
            retStr += "\n" + logExtra + "\n";
        }

        return retStr;
    }

    public void setAlwaysMatched(boolean alwaysMatched) {
        this.alwaysMatched = alwaysMatched;
    }

    public boolean isAlwaysMatched() {
        return alwaysMatched;
    }

    public int getGroupId() {
        return groupId;
    }

    public void setGroupId(int id) {
        groupId = id;
    }

    public int getDateIndex(LogFormat myLogFormat) {
        return myLogFormat.getDateColumnIndex();
    }

    public int getLogIndex(LogFormat myLogFormat) {
        return myLogFormat.getLogIndexColumnIndex();
    }

    public void setLogIndexNum(long logIndexNum) {
        this.logIndexNum = logIndexNum;
    }

    public long getLogIndexNum() {
        return logIndexNum;
    }

    public Date getLogDate() {
        return date;
    }

    public void appendLogExtra(int additionalLength) {
        readLength += additionalLength;
    }

    public boolean isProperlyFormed() {
        return properlyFormed;
    }

    public void setProperlyFormed(boolean properlyFormed) {
        this.properlyFormed = properlyFormed;
    }

    public RAFile getSrcFile(LogFormat myLogFormat) {
        return myLogFormat.getRandAccFile(logFileNum);
    }

    /** Returns log file index in array of log files for this log entry */
    public int getLogFileNum() {
        return logFileNum;
    }

    public long getFileOffset() {
        return fileOffset;
    }

    public int getReadLength() {
        return readLength;
    }

    public Date getDate() {
        return date;
    }

    public void setReadLength(int readLength) {
        this.readLength = readLength;
    }

    public void setDate(Date date) {
        this.date = date;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + (int) (logIndexNum ^ (logIndexNum >>> 32));
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        LogItem other = (LogItem) obj;
        if (logIndexNum != other.logIndexNum)
            return false;
        return true;
    }
}
