package com.vmware.vide.vlogbrowser.core.model;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Comparator;
import java.util.Date;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LogItemListRAFile implements ILogItemList {

    private static final Logger logger = LoggerFactory.getLogger(LogItemListRAFile.class);
    private static final int LOG_ITEM_BYTES = 38;

    // random access file that stores the log items
    private RandomAccessFile raFile;
    protected File file;
    private int size = 0;

    public LogItemListRAFile (File file) throws Exception {
        this.file = file;
        this.raFile = new RandomAccessFile(file, "rw");
        this.size = size();
    }

    /**
     * Add a logItem at the end of the list
     */
    @Override
    public boolean add(LogItem logItem) {
        return add(size, logItem, false);
    }

    private synchronized boolean add(long index, LogItem logItem, boolean update) {
        boolean wasAdded = false;
        try {
            raFile.seek(getOffset(index));
            writeLogItem(logItem);
            wasAdded = true;

            if (!update) {
                this.size++;
            }

        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        return wasAdded;
    }

    private synchronized void writeLogItem(LogItem logItem) throws IOException {
        raFile.writeBoolean(logItem.isAlwaysMatched()); // 1 byte
        raFile.writeBoolean(logItem.isProperlyFormed());// 1 byte
        raFile.writeInt(logItem.getGroupId());          // 4 bytes
        raFile.writeLong(logItem.getFileOffset());      // 8 bytes
        raFile.writeInt(logItem.getReadLength());       // 4 bytes
        raFile.writeLong(logItem.getDate().getTime());  // 8 bytes
        raFile.writeLong(logItem.getLogIndexNum());     // 8 bytes
        raFile.writeInt(logItem.getLogFileNum());       // 4 bytes
        // Writing a total of 38 bytes per LogItem
    }

    @Override
    public boolean addAll(ILogItemList logItemList) {
        boolean wasAdded = true;
        int logItemListSize = logItemList.size();
        for (int index = 0; (index < logItemListSize && wasAdded); index++) {
            LogItem logItem = logItemList.get(index);
            wasAdded = add(logItem);
        }
        return wasAdded;
    }

    @Override
    public synchronized int indexOf(LogItem logItem) {
        try {
            raFile.seek(0);
            int size = size();
            for (int index = 0; index < size; index++) {
                LogItem item = get(index);
                if (logItem.equals(item)) {
                    return index;
                }
            }
        } catch (IOException e) {
            return -1;
        }
        return -1;
    }

    @Override
    public synchronized boolean clear() {
        try {
            if (raFile != null) {
                raFile.close();
            }
            raFile = new RandomAccessFile(file, "rw");
            raFile.setLength(0);
            this.size = 0;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * returns null
     */
    @Override
    public synchronized LogItem get(int index) {
        LogItem result = null;
        try {
            raFile.seek(getOffset(index));
            result = readLogItem();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        return result;
    }

    private synchronized LogItem readLogItem() throws IOException {
        boolean alwaysMatched = raFile.readBoolean();
        boolean properlyFormed = raFile.readBoolean();
        int groupId = raFile.readInt();
        long fileOffset = raFile.readLong();
        int readLength = raFile.readInt();
        long time = raFile.readLong();
        long logIndexNum = raFile.readLong();
        int logFileNum = raFile.readInt();
        LogItem logItem = new LogItem(properlyFormed, fileOffset, readLength, new Date(time), logIndexNum, logFileNum);
        logItem.setAlwaysMatched(alwaysMatched);
        logItem.setGroupId(groupId);
        return logItem;
    }

    @Override
    public synchronized int size(){
        if (size == 0) {
            try {
                size = (int)((raFile.length())/LOG_ITEM_BYTES);
            } catch (Exception e) {
                logger.error("File " + file.getName() + " empty!", 0);
            }
        }
        return size;
    }
    
    @Override
    public synchronized void sort(Comparator<LogItem> comparator) {
        // TODO NILMAX Auto-generated method stub

    }

    @Override
    public synchronized boolean updateLogItem(long index, LogItem logItem) {
        return add(index, logItem, true);
    }
    
    private long getOffset(long index) {
        return (index * LOG_ITEM_BYTES);
    }
}