package com.vmware.vide.vlogbrowser.core.model;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.util.Comparator;
import java.util.Date;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LogItemListRAFileBuffer implements ILogItemList {

    private static final Logger logger = LoggerFactory.getLogger(LogItemListRAFileBuffer.class);
    private static final int LOG_ITEM_BYTES = 30;
    private static final int NUM_LOG_ITEM = 100;

    // random access file that stores the log items
    private RandomAccessFile raFile;
    protected File file;
    private int size = 0;
    private long readBase = -1;

    private ByteBuffer buffer =  ByteBuffer.allocate(LOG_ITEM_BYTES * NUM_LOG_ITEM);

    public LogItemListRAFileBuffer (File file) throws Exception {
        this.file = file;
        this.raFile = new RandomAccessFile(file, "rw");
        this.size = size();
    }

    /**
     * Add a logItem at the end of the list
     */
    @Override
    public boolean add(LogItem logItem) {
        return add(size, logItem, false);
    }

    private synchronized boolean add(long index, LogItem logItem, boolean update) {
        boolean wasAdded = false;
        try {
            if (readBase != -1) {
                buffer.clear();
                readBase = -1;
            }
            if (!buffer.hasRemaining()) {
                flushBuffer();
            }
            int pos = (int)(index % NUM_LOG_ITEM);
            buffer.position((int)getOffset(pos));
            writeLogItem(logItem);
            wasAdded = true;

            if (!update) {
                this.size++;
            }

        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        return wasAdded;
    }

    private synchronized void writeLogItem(LogItem logItem) throws IOException {
        buffer.put(booleanToByte(logItem.isAlwaysMatched())); // 1 byte
        buffer.put(booleanToByte(logItem.isProperlyFormed()));// 1 byte
        buffer.putInt(logItem.getGroupId());          // 4 bytes
        buffer.putLong(logItem.getFileOffset());      // 8 bytes
        buffer.putInt(logItem.getReadLength());       // 4 bytes
        buffer.putLong(logItem.getDate().getTime());  // 8 bytes
        buffer.putInt(logItem.getLogFileNum());       // 4 bytes
        // Writing a total of 30 bytes per LogItem
    }

    private void flushBuffer() throws IOException {
        int bufferRemaining = (int)(buffer.capacity() - buffer.position())/LOG_ITEM_BYTES;
        int pos = size + bufferRemaining - NUM_LOG_ITEM;
        raFile.seek(getOffset(pos));
        raFile.write(buffer.array(), 0, buffer.position());
        buffer.clear();
    }

    private byte booleanToByte(boolean value) {
        if (value) {
            return (byte) 1;
        }
        return (byte) 0;
    }

    private boolean byteToBoolean (byte value) {
        return value == (byte) 1;
    }

    @Override
    public boolean addAll(ILogItemList logItemList) {
        boolean wasAdded = true;
        int logItemListSize = logItemList.size();
        for (int index = 0; (index < logItemListSize && wasAdded); index++) {
            LogItem logItem = logItemList.get(index);
            wasAdded = add(logItem);
        }
        return wasAdded;
    }

    @Override
    public synchronized int indexOf(LogItem logItem) {
        try {
            raFile.seek(0);
            for (int index = 0; index < size; index++) {
                LogItem item = get(index);
                if (logItem.equals(item)) {
                    return index;
                }
            }
        } catch (IOException e) {
            return -1;
        }
        return -1;
    }

    @Override
    public synchronized boolean clear() {
        try {
            if (raFile != null) {
                raFile.close();
            }
            raFile = new RandomAccessFile(file, "rw");
            raFile.setLength(0);
            this.size = 0;
            buffer.clear();
            readBase = -1;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * returns null
     */
    @Override
    public synchronized LogItem get(int index) {
        LogItem result = null;
        try {
            if (index < 0 || index >= size) {
                throw new IndexOutOfBoundsException();
            }
            if (readBase == -1  && buffer.position() != 0) {
                flushBuffer();
            }
            int base = (int)(index / NUM_LOG_ITEM) * NUM_LOG_ITEM;
            if (readBase != base) {
                raFile.seek(getOffset(base));
                raFile.read(buffer.array());
                readBase = base;
            }
            int pos = (int)(index % NUM_LOG_ITEM);
            buffer.position((int)getOffset(pos));
            result = readLogItem();
            result.setLogIndexNum(index);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        return result;
    }

    private synchronized LogItem readLogItem() throws IOException {
        boolean alwaysMatched = byteToBoolean(buffer.get());
        boolean properlyFormed = byteToBoolean(buffer.get());
        int groupId = buffer.getInt();
        long fileOffset = buffer.getLong();
        int readLength = buffer.getInt();
        long time = buffer.getLong();
        int logFileNum = buffer.getInt();
        LogItem logItem = new LogItem(properlyFormed, fileOffset, readLength,
                new Date(time), -1, logFileNum);
        logItem.setAlwaysMatched(alwaysMatched);
        logItem.setGroupId(groupId);
        return logItem;
    }

    @Override
    public synchronized int size(){
        if (size == 0) {
            try {
                size = (int)((raFile.length())/LOG_ITEM_BYTES);
            } catch (Exception e) {
                logger.error("File " + file.getName() + " empty!", 0);
            }
        }
        return size;
    }

    @Override
    public synchronized void sort(Comparator<LogItem> comparator) {
        // TODO Auto-generated method stub
    }

    @Override
    public synchronized boolean updateLogItem(long index, LogItem logItem) {
        return add(index, logItem, true);
    }

    private long getOffset(long index) {
        return (index * LOG_ITEM_BYTES);
    }
}
