/***********************************************************
 * Copyright 2010 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core.model;

import java.net.URL;
import java.util.List;

/**
 * The LogxFile class represents a .logx file. It is used by the wizard to represent a
 * generic/template .logx file (originals located in the .../templates/ directory),
 * and it is also used to represent an actual .logx file that is customized by the user
 * (in the wizard) to represent an actual .logx file in some project container.
 */
public class LogxFile {

    private boolean createThisFile;     // for "Enable?" column on the wizard table
    private String logFormatName;
    private String logCategory;
    private String logxFileName;
    private boolean openEditor;     // for "Open?" column on the wizard table (Finish screen)
    private String srcHost;
    private String archivePath;
    private LogSystemType systemType;
    private URL templateURL;

    private List<String> fileRegexes;

    public LogxFile(String logFormatName,
                    String category,
                    URL templateURL, LogSystemType systemType)
    {
        this.logFormatName = logFormatName;
        this.systemType = systemType;
        this.logCategory = category;
        this.templateURL = templateURL;
    }

    public void setCreateThisFile(boolean createThisFile) {
        this.createThisFile = createThisFile;
    }

    public boolean isCreateThisFile() {
        return createThisFile;
    }

    public String getLogFormatName() {
        return logFormatName;
    }

    public String getCategory() {
        return logCategory;
    }

    public void setLogxFileName(String logxFileName) {
        this.logxFileName = logxFileName;
    }

    public String getLogxFileName() {
        return logxFileName;
    }

    public void setOpenEditor(boolean openEditor) {
        this.openEditor = openEditor;
    }

    public boolean isOpenEditor() {
        return openEditor;
    }

    public void setSrcHost(String srcHost) {
        this.srcHost = srcHost;
    }

    public String getSrcHost() {
        return srcHost;
    }

    public void setArchivePath(String archivePath) {
        this.archivePath = archivePath;
    }

    public String getArchivePath() {
        return archivePath;
    }

    public void setFileRegexes(List<String> fileRegexes) {
        this.fileRegexes = fileRegexes;
    }

    public List<String> getFileRegexes() {
        return fileRegexes;
    }

    public LogSystemType getSystemType() {
        return systemType;
    }

    public String toSummaryString() {
        String retStr = srcHost + " (" + logFormatName + ")";
        if ((archivePath != null) && (archivePath.length() > 0)) {
            retStr += " (vm-support)";
        }
        return retStr;
    }

    /**
     * This is used by the wizard when it calls the List<LogxFile>#contains method. If these
     * fields match, the LogxFile object is similar enough, and should not be added again to the
     * table on the 3rd wizard screen.
     */
    @Override
    public boolean equals(Object obj) {
        if (obj instanceof LogxFile) {
            LogxFile otherLxFile = (LogxFile) obj;
            if ((npeSafeEquals(this.srcHost, otherLxFile.srcHost)) &&
               (npeSafeEquals(this.logFormatName, otherLxFile.logFormatName)) &&
               (npeSafeEquals(this.getSystemType(), otherLxFile.getSystemType())) &&
               (npeSafeEquals(this.archivePath, otherLxFile.archivePath))) {
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    private boolean npeSafeEquals(Object thisObj, Object otherObj) {
        if (thisObj == null) {
            if (otherObj != null) {
                return false;
            }
            else {
                return true;
            }
        } else {
            return thisObj.equals(otherObj);
        }
    }
    /** Returns filepath for template file, which this logx was created from */
    public URL getTemplateURL() {
        return templateURL;
    }
}
