/***********************************************************
 * Copyright 2009 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core.parser;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.util.HashMap;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * The AutoDetectLogType class takes a log file and a List of LogFormats and tries to
 * determine which LogFormat matches the log file. It first tries to compare the name of the
 * log file to a list of known log format names defined (with RegEx) in each LogFormat. If
 * no match is found, it parses up to the first 50K bytes in each log file and tries to match
 * any log entries found in the file to the columns defined in a LogFormat. The first LogFormat
 * to reach 50 perfectly matching lines in the log file "wins" for that file, and is returned
 * as the matching LogFormat for that file. This class is used by the LogBrowserWizardPage
 * class, such that when the user chooses a log file to analyze, the wizard screen can detect
 * the log format type and preset the Log Format drop-down list to that value.
 */
public class AutoDetectLogType {

    private String filename;
    private InputStream logInputStream;
    private List<LogFormat> knownLogFormats;
    HashMap<LogFormat, Integer> logFormatScores;
    public static int SCORE_LIMIT = 50;
    public static int READ_BUFFER_LIMIT = 50000;

    public static final String VMSUPPORT_REGEX = "esx.+\\.tgz$";
    public static final String VMSUPPORT_TYPE = "vm-support_log";

    public static String newline = System.getProperty("line.separator");

    public AutoDetectLogType(String filename,
                             InputStream logInputStream,
                             List<LogFormat> knownLogFormats) {
        this.filename = filename;
        this.logInputStream = logInputStream;
        this.knownLogFormats = knownLogFormats;
        this.logFormatScores = new HashMap<LogFormat, Integer>();
    }

    public AutoDetectLogType(File logfile,
                             List<LogFormat> knownLogFormats) throws FileNotFoundException {
        this.filename = logfile.getName();
        this.knownLogFormats = knownLogFormats;
        logInputStream = new FileInputStream(logfile);
        this.logFormatScores = new HashMap<LogFormat, Integer>();
    }

    public static boolean storedInsideVMSuppLog(String fileStr) {
        if ((fileStr != null) && (fileStr.length() > 0)){
            Pattern nameRegexPattern = Pattern.compile(VMSUPPORT_REGEX);
            return nameRegexPattern.matcher(fileStr).find();
        } else {
            return false;
        }
    }

    public String getLogType() throws IOException {

        // First check to see if this is a vm-support archive
        if (storedInsideVMSuppLog(filename)) {
            return VMSUPPORT_TYPE;
        }
        // Keep score; if more than one LogFormat filename matches, tie-break from InputStream
        computeScoresFromName();
        if (logFormatScores.size() == 1) {      // exactly one match, so return this one
            return logFormatScores.keySet().toArray(new LogFormat[1])[0].getName();
        } else {
            computeScoresFromStream();
            LogFormat bestMatchingFormat = null;
            Integer highestScore = 0;
            for (LogFormat logFormat : logFormatScores.keySet()) {
                int currentScore = logFormatScores.get(logFormat);
                if (currentScore > highestScore) {
                    highestScore = currentScore;
                    bestMatchingFormat = logFormat;
                }
            }
            if (bestMatchingFormat != null) {
                return bestMatchingFormat.getName();
            } else {
                return "";
            }
        }

    }

    private void computeScoresFromName() {

        for (LogFormat logFormat : knownLogFormats) {
            List<String> filenames = logFormat.getKnownFilenames();
            for (String regexFilename : filenames) {
                Pattern regexMatch = Pattern.compile(regexFilename);
                Matcher matcher = regexMatch.matcher(filename);
                if (matcher.find()) {
                    logFormatScores.put(logFormat, 1);
                    break;
                }
            }
        }
    }

    private void computeScoresFromStream() throws IOException {

        InputStreamReader logReader = new InputStreamReader(logInputStream);
        BufferedReader buffRdr = new BufferedReader(logReader, READ_BUFFER_LIMIT);
        char[] myCharArr = new char[READ_BUFFER_LIMIT];
        int fileLen = buffRdr.read(myCharArr);
        if (fileLen <= 0) {
            return;
        }
        String myStr = new String(myCharArr);
        myStr = myStr.substring(0, fileLen);

        StringReader strRdr = new StringReader(myStr);
        BufferedReader buffStrRdr = new BufferedReader(strRdr);
        String logline;
        boolean scoreLimitReached = false;
        while (((logline = buffStrRdr.readLine()) != null) && (!scoreLimitReached)) {
            for (LogFormat logFormat : knownLogFormats) {
                if (logFormat.matchesCompletely(logline)) {
                    Integer previousScore = logFormatScores.get(logFormat);
                    int currentScore = (previousScore != null) ? previousScore + 1 : 1;
                    logFormatScores.put(logFormat, currentScore);
                    if (currentScore == SCORE_LIMIT) {
                        scoreLimitReached = true;
                        break;
                    }
                }
            }
        }
        buffRdr.close();
    }

}
