/***********************************************************
 * Copyright 2010 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core.parser;

import java.io.IOException;

import com.vmware.vide.vlogbrowser.core.model.LogFilter;
import com.vmware.vide.vlogbrowser.core.model.LogItem;

/**
 * The ComplexMatcher class is used (via the Matcher<LogItem> interface) to handle
 * filtering of LogItems from the Log Table View. A LogFilter object, which is configured by the
 * Filter Dialog, serves as the actual matching engine. This class uses LogFilter to check the
 * String that comprises the LogItem object to see if that String matches the configured LogFilter
 * criteria. In some cases, the match checking is short-circuited. For example,
 * the user may use the "Include # Surrounding Lines" drop-down box. In this case, surrounding
 * LogItems will be set to "Always Match" by the updateFilter() method in the LogTableViewer
 * class. This class inherits the "matches()" method that enables short circuit matching.
 *
 * This class, along with LogItemMatcher, both implement the MultiMatcher interface which extends
 * Matcher interface. LogTableViewer will be given an object of the type
 * MultiMatcher, which is implemented by the LogItemMatcher class for single
 * keyword matches, and which is implemented by this class for more complex filtering operations.
 */
public class ComplexMatcher implements MultiMatcher {

    private LogFilter logFilter;
    private LogFormat logFormat;

    public ComplexMatcher(LogFilter logFilter, LogFormat logFormat) {
        this.logFilter = logFilter;
        this.logFormat = logFormat;
    }

    @Override
    public boolean matches(LogItem logItem) {
        if (logItem.isAlwaysMatched()) {
            return true;
        }
        return matchesSomeField(logItem);
    }

    @Override
    public boolean matchesSomeField(LogItem logItem) {
        try {
            return logFilter.evaluateFilterMatrix(new String(logItem.getLogEntryBytes(logFormat)));
        } catch (IOException ioe) {
            ioe.printStackTrace();
        }
        return false;
    }

}
