/***********************************************************
 * Copyright 2009 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core.parser;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.vmware.vide.vlogbrowser.core.consts.LogBrowserConsts;
import com.vmware.vide.vlogbrowser.core.fileops.LogFileManager;
import com.vmware.vide.vlogbrowser.core.fileops.LogFileUtils;
import com.vmware.vide.vlogbrowser.core.model.FilterItem;
import com.vmware.vide.vlogbrowser.core.model.LogFilter;
import com.vmware.vide.vlogbrowser.core.model.LogSystemType;
import com.vmware.vide.vlogbrowser.core.model.LogSystemType.SortMethod;
import com.vmware.vide.vlogbrowser.service.LogFileAnalyzer;

/**
 * The ConfigParser class reads in an XML configuration file that describes the format of a
 * a type of log and stores that XML internally as a DOM object. It also produces a LogFormat
 * object to represent that log format. The XML config file is the file that it is loaded into
 * the Eclipse "Editor" that makes up LogBrowser. If the XML config file content is read from
 * a template by the "Create new..." LogBrowser wizard, it is missing the log file path element.
 * For that reason, a setLogFilePath() method is provided to add that element to the DOM after
 * an XML template file has been read in. If the user is reading a LogBrowser config file that
 * has already been created, the LogFilePath element is already present, referring to the
 * log file on disk that is to be parsed and displayed. Please see the LogFormat class
 * for more information.
 */
public class ConfigParser {

    private File configFile;
    private DocumentBuilderFactory docBldrFactory;
    private DocumentBuilder docBldr;
    private Document configDoc;
    private LogSystemType fSystemType;
    
    private static final Logger logger = LoggerFactory.getLogger(ConfigParser.class);

    public ConfigParser(InputStream inputStream) throws Exception {
        this(null, inputStream, true);
    }

    public ConfigParser(InputStream inputStream, boolean requireID) throws Exception {
        this(null, inputStream, requireID);
    }

    public ConfigParser(File configFile, InputStream inputStream) throws Exception {
        this(configFile, inputStream, true);
    }

    public ConfigParser(File configFile, InputStream inputStream, boolean requireID) throws Exception {
        this.configFile = configFile;
        docBldrFactory = DocumentBuilderFactory.newInstance();
        try {
            docBldr = docBldrFactory.newDocumentBuilder();
        } catch (ParserConfigurationException e) {
            logger.debug(e.getMessage(), e);
        }
        try {
            configDoc = docBldr.parse(inputStream);
        } catch (SAXException e) {
            throw new Exception("LogFormat configuration could not be parsed. " +
                    "The configuration must be properly formatted XML.", e);
        } catch (IOException e) {
            throw new Exception("LogFormat configuration could not be read.", e);
        } catch (IllegalArgumentException e) {
            throw new Exception("LogFormat configuration must not be null", e);
        }
        if (requireID) {
            try {
                getUniqueID();
            } catch (Exception e) {
                updateUniqueID();
            }
        }
    }

    public LogFormat getLogFormat() throws Exception {
        LogFormat logFormat = null;

        XPath xpath = XPathFactory.newInstance().newXPath();
        int numLogFormats = 0;
        try {
            numLogFormats = Integer.parseInt(xpath.evaluate("count(/root/LogFormat)", configDoc));
            if (numLogFormats == 0) {
                throw new Exception("No LogFormat nodes found in XML tree.");
            }
        } catch (Exception e) {
            throw new Exception("Could not parse XML LogFormat configuration file " +
                    " successfully to determine number of LogFormat nodes." + e.getMessage());
        }

        try {
            String logFormatName = xpath.evaluate("/root/LogFormat[1]/Name/text()", configDoc);

            logFormat = new LogFormat(logFormatName);

            int formatIndex = 1;
            int numFields = getFields(xpath, formatIndex, logFormatName, logFormat);
            if (numFields == 0) {
                throw new Exception("There should be at least one Field defined in " + logFormatName);
            }

            int numColumns = getColumns(xpath, formatIndex, logFormatName, logFormat);
            if (numColumns == 0) {
                throw new Exception("There should be at least one Column defined in " + logFormatName);
            }

            int numFilenames = getKnownSystypes(xpath, formatIndex, logFormatName, logFormat);
            if (numFilenames == 0) {
                throw new Exception("There should be at least one KnownFilename defined in " + logFormatName);
            }
            logFormat.setDateDelta(getDateDelta());
            setSizeLimit(logFormat);
        }
        catch (Exception e) {
            e.printStackTrace();
            throw new Exception("Could not parse LogFormat file.", e);
        }

        return logFormat;
    }

    /** Parse and return system type and file info from log file */
    public LogSystemType getSystemType() {
        // parse LogSystemType on first method call
        if (fSystemType == null) {
            Element format = (Element)configDoc.getElementsByTagName("LogFormat").item(0);
            Element sysTypeElement = (Element)format.getElementsByTagName("SystemType").item(0);
            String liveLog = sysTypeElement.getAttribute("live-log");
            String regex = sysTypeElement.getAttribute("file");
            String dir = sysTypeElement.getAttribute("location");
            String system = sysTypeElement.getAttribute("system");
            String sysVer = sysTypeElement.getAttribute("version");
            String descending = sysTypeElement.getAttribute("number-descending");

            String sortStr = sysTypeElement.getAttribute("sort");
            SortMethod sort = SortMethod.Filenum;
            if (sortStr.equals("filename"))
                sort = SortMethod.Filename;
            if (sortStr.equals("time"))
                sort = SortMethod.Time;

            boolean isDescending = descending != null &&
                (descending.equalsIgnoreCase("true") || descending.equalsIgnoreCase("yes"));
            fSystemType = new LogSystemType(system, sysVer, dir, regex, liveLog, isDescending, sort);
        }
        return fSystemType;
    }

    private int getFields(XPath xpath, int logFormatIndex,
            String logFormatName, LogFormat logFormatObj)
            throws Exception {
        int numFields = Integer.parseInt(
                xpath.evaluate("count(/root/LogFormat[" +
                               Integer.toString(logFormatIndex) +
                               "]/Fields/Field)",
                               configDoc));
        String logFormatNum = Integer.toString(logFormatIndex);
        for (int j = 1; j <= numFields; j++) {
            String fieldNum = Integer.toString(j);
            String name = xpath.evaluate("/root/LogFormat[" + logFormatNum +
                    "]/Fields/Field[" + fieldNum + "]/Name/text()", configDoc);
            String lowerBound = xpath.evaluate("/root/LogFormat[" + logFormatNum +
                    "]/Fields/Field[" + fieldNum + "]/LowerBound/text()", configDoc);
            String upperBound = xpath.evaluate("/root/LogFormat[" + logFormatNum +
                    "]/Fields/Field[" + fieldNum + "]/UpperBound/text()", configDoc);
            String regexMatch = xpath.evaluate("/root/LogFormat[" + logFormatNum +
                    "]/Fields/Field[" + fieldNum + "]/RegexMatch/text()", configDoc);
            String regexCaptureGroup = xpath.evaluate("/root/LogFormat[" + logFormatNum +
                    "]/Fields/Field[" + fieldNum + "]/RegexCaptureGroup/text()", configDoc);
            logFormatObj.addField(
                    name, lowerBound, upperBound, regexMatch, regexCaptureGroup);
        }
        return numFields;
    }

    private int getColumns(XPath xpath, int logFormatIndex,
            String logFormatName, LogFormat logFormatObj) throws Exception {
        int numColumns = Integer.parseInt(xpath.evaluate("count(/root/LogFormat[" +
                Integer.toString(logFormatIndex) + "]/Columns/Column)", configDoc));
        String logFormatNum = Integer.toString(logFormatIndex);
        for (int j = 1; j <= numColumns; j++) {
            String columnNum = Integer.toString(j);
            String columnName = xpath.evaluate("/root/LogFormat[" + logFormatNum +
                    "]/Columns/Column[" + columnNum + "]/Name/text()", configDoc);
            String isDefault = xpath.evaluate("/root/LogFormat[" + logFormatNum +
                    "]/Columns/Column[" + columnNum + "]/Default/text()", configDoc);
            String widthWeight = xpath.evaluate("/root/LogFormat[" + logFormatNum +
                    "]/Columns/Column[" + columnNum + "]/WidthWeight/text()", configDoc);
            String columnType = xpath.evaluate("/root/LogFormat[" + logFormatNum +
                    "]/Columns/Column[" + columnNum + "]/ColumnType/text()", configDoc);
            String dateFormat = xpath.evaluate("/root/LogFormat[" + logFormatNum +
                    "]/Columns/Column[" + columnNum + "]/DateFormat/text()", configDoc);
            String fieldGlue = xpath.evaluate("/root/LogFormat[" + logFormatNum +
                    "]/Columns/Column[" + columnNum + "]/FieldGlue/text()", configDoc);
            logFormatObj.addColumn(columnName,
                                   isDefault,
                                   widthWeight,
                                   columnType,
                                   dateFormat,
                                   fieldGlue);
            int numFieldsInColumn = Integer.parseInt(xpath.evaluate("count(/root/LogFormat[" +
                    Integer.toString(logFormatIndex) + "]/Columns/Column[" +
                    Integer.toString(j) + "]/AssignFields/FieldName)", configDoc));
            if ((numFieldsInColumn == 0) &&
                    (!columnType.equals(LogFormat.ColumnType.LOGINDEX.toString()))) {
                throw new Exception(" There should be at least one Field assigned to Column "
                                    + columnName);
            }
            for (int k = 1; k <= numFieldsInColumn; k++) {
                String fieldName = xpath.evaluate("/root/LogFormat[" +
                        logFormatNum + "]/Columns/Column[" +
                        columnNum + "]/AssignFields/FieldName[" +
                        Integer.toString(k) + "]/text()", configDoc);
                logFormatObj.addFieldToColumn(fieldName, columnName);
            }
        }
        return numColumns;
    }

    private int getKnownSystypes(XPath xpath, int logFormatIndex,
            String logFormatName, LogFormat logFormatObj) throws Exception {
        int numFilenames = Integer.parseInt(
                xpath.evaluate("count(/root/LogFormat[" +
                               Integer.toString(logFormatIndex) +
                               "]/SystemType)",
                               configDoc));
        return numFilenames;
    }

    public String getLogFilePath() throws Exception {
        XPath xpath = XPathFactory.newInstance().newXPath();
        String logFilePath = null;
        try {
            logFilePath = xpath.evaluate("/root/InputLogFile/text()", configDoc);
            if ((logFilePath == null) || (logFilePath.length() == 0)) {
                throw new Exception();
            }
        } catch (Exception e) {
            throw new Exception(" Log file name is missing from the configuration.");
        }
        return logFilePath;
    }

    public void setLogFilePath(String logFilePath) {
        XmlDocUtils.replaceXmlElement(configDoc, "InputLogFile", logFilePath);
    }

    /**
     * Convert the DOM object into an XML string
     */
    public String getXmlFromDocument() {
        return XmlDocUtils.getXmlStringFromDocument(configDoc);
    }

    public String[] getSourceIDs() throws Exception {
        return XmlDocUtils.getElements(configDoc, "/root/SourceLogFiles/Source", "ID");
    }

    public String[] getFileRegExesInSrc(String id) throws Exception {
        return XmlDocUtils.getElements(configDoc, "/root/SourceLogFiles/Source[ID='" + id + "']/FileRegEx", "");
    }

    public String getConfigFilePath() {
        return configFile.getAbsolutePath();
    }

    public String updateUniqueID() throws Exception {
        XmlDocUtils.replaceXmlElement(configDoc, "logx_ID", Double.toHexString(Math.random()));
        return XmlDocUtils.getElement(configDoc, "/root/logx_ID");
    }

    public String getUniqueID() throws Exception {
        return XmlDocUtils.getElement(configDoc, "/root/logx_ID");
    }

    public void updateDateDelta(long dateDelta) throws Exception {
        XmlDocUtils.replaceXmlElement(configDoc,
                                      "DateDelta",
                                      Long.toString(dateDelta));
    }

    public long getDateDelta() {
        try {
            String myStr = XmlDocUtils.getElement(configDoc, "/root/DateDelta");
            return Long.parseLong(myStr);
        } catch (Exception e) {
            return 0L;
        }
    }

    public void setSizeLimit(LogFormat fmt) {
        try {
            String myStr = XmlDocUtils.getElement(configDoc, "/root/LogFormat/SizeLimit");
            if (myStr != null && !myStr.isEmpty()) {
                fmt.setSizeLimitMB(Integer.parseInt(myStr));
            }
        } catch (Exception e) {
        }
    }

    public String getLogFormatName() throws Exception {
        return XmlDocUtils.getElement(configDoc, "/root/LogFormat/Name");
    }

    public String getCategory() throws Exception {
        return XmlDocUtils.getElement(configDoc, "/root/LogFormat/Category");
    }

    public void addSrcLogFileNode(String srcHostname,
                                  String parentDir,
                                  String archivePath,
                                  String[] fileRegExArr) throws Exception {

        if (LogFileManager.LOCALHOSTURL.equalsIgnoreCase(srcHostname) ||
                LogFileManager.LOCALHOST.equalsIgnoreCase(srcHostname)) {
            srcHostname = LogFileManager.LOCALHOST;
        }
        if ((archivePath != null) && (archivePath.length() > 0)) {
            archivePath = ";" + archivePath;
        } else {
            archivePath = "";
        }
        String idStr = srcHostname + archivePath + ";" + parentDir;

        XPath xpath = XPathFactory.newInstance().newXPath();
        Object sourceNodeObj = null;
        try {       // try to find an existing <Source> node with this ID element
            sourceNodeObj = xpath.evaluate("/root/SourceLogFiles/Source[ID='" + idStr + "']",
                                           configDoc,
                                           XPathConstants.NODE);
        } catch (Exception e) {
            throw new Exception("There was an XML parsing problem while locating " + idStr);
        }
        Element sourceElmt = null;
        if (sourceNodeObj == null) {    // if above <Source> node not found, create new one
            Element idElmt = configDoc.createElement("ID");
            idElmt.appendChild(configDoc.createTextNode(idStr));
            sourceElmt = configDoc.createElement("Source");
            sourceElmt.appendChild(idElmt);
        } else {
            sourceElmt = (Element) sourceNodeObj;
        }

        for (String fileREStr : fileRegExArr) {
            String testStr = null;
            try {       // search for existing fileRegEx String (don't duplicate)
                String mySearchStr = "/root/SourceLogFiles/Source[ID='" + idStr +
                    "' and FileRegEx='" + fileREStr + "']" + "/FileRegEx[text()='" +
                    fileREStr + "']/text()";
                testStr = xpath.evaluate(mySearchStr, configDoc);
            } catch (Exception e) {
                throw new Exception("There was an XML parsing problem while locating " + testStr);
            }
            if ((testStr == null) || (testStr.trim().length() == 0)) {  // if search not found...
                Element fileREElmt = configDoc.createElement("FileRegEx");
                fileREElmt.appendChild(configDoc.createTextNode(fileREStr));
                sourceElmt.appendChild(fileREElmt);
            }
        }

        Node rootNode = configDoc.getFirstChild();

        Element sourceLogFilesElmt = null;
        NodeList nodeList = rootNode.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            if (nodeList.item(i).getNodeName().equals("SourceLogFiles")) {
                sourceLogFilesElmt = (Element) nodeList.item(i);
                break;
            }
        }
        if (sourceLogFilesElmt == null) {
            sourceLogFilesElmt = configDoc.createElement("SourceLogFiles");
            rootNode.appendChild(sourceLogFilesElmt);
        }

        sourceLogFilesElmt.appendChild(sourceElmt);
    }

    public void addArchiveSrcLogElement(String srcHostChoice, String archivePath) throws Exception {
        String parentDir = fSystemType.getDir();
        String[] regExArr = new String[]{fSystemType.getFileRegex()};
        addSrcLogFileNode(srcHostChoice, parentDir, archivePath, regExArr);
    }

    public void addRemoteSrcLogElement(String srcHostChoice) throws Exception {
        String parentDir = fSystemType.getDir();
        String[] regExArr = new String[]{fSystemType.getFileRegex()};
        addSrcLogFileNode(srcHostChoice, parentDir, "", regExArr);
    }

    public void addLocalSrcLogElements(String logFilePaths) throws Exception {
        if ((logFilePaths == null) || (logFilePaths.length() == 0)) {
            return;
        }
        String[] logFileArr = logFilePaths.split(";");
        if (logFileArr.length == 0) {
            return;
        }
        String parentDir = LogFileUtils.getParentDir(logFileArr[0]);
        List<String> logFileList = new ArrayList<String>();
        for (String logFilePath : logFileArr) {
            logFileList.add(LogFileUtils.getFileName(logFilePath) + "$");
        }
        addSrcLogFileNode(LogFileManager.LOCALHOST,
                          parentDir,
                          "",
                          logFileList.toArray(new String[logFileList.size()]));
    }

    /**
     *  Take a named LogFilter object and put it in the XML configuration object so that it can
     *  be written to the XML configuration file (.logx file) by
     *  LogBrowserEditor.overwriteTextInEditor(), thus saving to disk. The XML format is:
     *  <root>
     *  <!-- Other config file stuff like log format, etc. -->
     *      <Filters>
     *          <Filter>
     *              <Name><![CDATA[]]></Name>
     *              <FilterRows>
     *                  <FilterRow>
     *                      <FilterItems>
     *                          <FilterItem>
     *                              <BooleanConnector></BooleanConnector>
     *                              <NOT></NOT>
     *                              <Keyword><![CDATA[]]></Keyword>
     *                              <RegEx></RegEx>
     *                          </FilterItem>
     *                          <!-- Zero or more FilterItem nodes (columns) -->
     *                      </FilterItems>
     *                  </FilterRow>
     *                  <!-- Zero or more FilterRow nodes (rows) -->
     *              </FilterRows>
     *              <NumSurroundingLines></NumSurroundingLines>
     *          <Filter>
     *          <!-- Zero or more Filter nodes (named filters) -->
     *      </Filters>
     *  </root>
     */
    public void addLogFilterNode(LogFilter logFilter) throws Exception {

        Element filtersElmt = getFiltersElement();
        if (filtersElmt == null) {
            filtersElmt = configDoc.createElement("Filters");
            Node rootNode = configDoc.getFirstChild();
            rootNode.appendChild(filtersElmt);
        }

        deleteLogFilter(logFilter.getName(), filtersElmt);    // remove and replace
        buildLogFilterNode(logFilter, filtersElmt);

    }

    private Element getFiltersElement() {
        Node rootNode = configDoc.getFirstChild();      // <root>

        // Get the <Filters> element
        Element filtersElmt = null;
        NodeList nodeList = rootNode.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            if (nodeList.item(i).getNodeName().equals("Filters")) {
                filtersElmt = (Element) nodeList.item(i);
                break;
            }
        }
        return filtersElmt;
    }

    private void deleteLogFilter(String lfName, Element filtersElmt) {
        Node filter = null;
        boolean found = false;
        NodeList filtersChildren = filtersElmt.getChildNodes();
        for (int i = 0; i < filtersChildren.getLength(); i++) {
            if (filtersChildren.item(i).getNodeName().equals("Filter")) {
                filter = filtersChildren.item(i);
                NodeList filterChildren = filter.getChildNodes();
                for (int j = 0; j < filterChildren.getLength(); j++) {
                    Node filterChild = filterChildren.item(j);
                    if (filterChild.getNodeName().equals("Name")) {
                        Node cdataName = filterChild.getFirstChild();
                        if ( (cdataName != null) &&
                             (cdataName.getNodeType() == Node.CDATA_SECTION_NODE) &&
                             (cdataName.getTextContent().equals(lfName))) {
                            found = true;
                            break;
                        }
                    }
                }
            }
            if (found) {
                break;
            }
        }
        if (found) {
            filtersElmt.removeChild(filter);
        }
    }

    public void deleteLogFilter(String logFilterName) {
        Element filtersElmt = getFiltersElement();
        if (filtersElmt == null) {
            return;
        }
        deleteLogFilter(logFilterName, filtersElmt);
    }

    /*
     * See Javadoc on addLogFilterNode() for XML format
     */
    private void buildLogFilterNode(LogFilter logFilter, Element filtersElmt) throws Exception {

        Element filterElmt = configDoc.createElement("Filter");
        Element filterNameElmt = configDoc.createElement("Name");
        filterNameElmt.appendChild(configDoc.createCDATASection(logFilter.getName()));
        filterElmt.appendChild(filterNameElmt);
        Element filterRowsElmt = configDoc.createElement("FilterRows");
        for (int rowIdx = 0; rowIdx < logFilter.getNumRows(); rowIdx++) {
            Element filterRowElmt = configDoc.createElement("FilterRow");
            Element filterItemsElmt = configDoc.createElement("FilterItems");
            for (int colIdx = 0; colIdx < logFilter.getNumColumns(rowIdx); colIdx++) {
                FilterItem filterItem = logFilter.getFilterItem(rowIdx, colIdx);

                Element boolConnElmt = configDoc.createElement("BooleanConnector");
                boolConnElmt.appendChild(configDoc.createTextNode(filterItem.getBoolConnector().toString()));

                Element notElmt = configDoc.createElement("NOT");
                notElmt.appendChild(configDoc.createTextNode((filterItem.isNegateFlagSet()) ? "True" : "False"));

                Element keywordElmt = configDoc.createElement("Keyword");
                keywordElmt.appendChild(configDoc.createCDATASection(filterItem.getKeyword()));

                Element regExElmt = configDoc.createElement("RegEx");
                regExElmt.appendChild(configDoc.createTextNode((filterItem.isRegexFlagSet()) ? "True" : "False"));

                Element filterItemElmt = configDoc.createElement("FilterItem");
                filterItemElmt.appendChild(boolConnElmt);
                filterItemElmt.appendChild(notElmt);
                filterItemElmt.appendChild(keywordElmt);
                filterItemElmt.appendChild(regExElmt);
                filterItemsElmt.appendChild(filterItemElmt);
            }
            filterRowElmt.appendChild(filterItemsElmt);
            filterRowsElmt.appendChild(filterRowElmt);
        }
        filterElmt.appendChild(filterRowsElmt);
        Element numSurrLinesElmt = configDoc.createElement("NumSurroundingLines");
        numSurrLinesElmt.appendChild(configDoc.createTextNode(Integer.toString(logFilter.getNumSurroundingLines())));
        filterElmt.appendChild(numSurrLinesElmt);

        filtersElmt.appendChild(filterElmt);

    }

    /**
     * Read and return the named log filters stored in the XML configuration object (which
     * was originally parsed from and represents the .logx file)
     */
    public Map<String, LogFilter> getLogFilters() throws Exception {
        HashMap<String, LogFilter> namedLogFilters = new HashMap<String, LogFilter>();

        Node rootNode = configDoc.getFirstChild();      // <root>

        Element filtersElmt = null;     // Get the <Filters> element, or return empty Hashmap
        NodeList nodeList = rootNode.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            if (nodeList.item(i).getNodeName().equals("Filters")) {
                filtersElmt = (Element) nodeList.item(i);
                break;
            }
        }
        if (filtersElmt == null) {
            return namedLogFilters;
        }

        NodeList filtersChildren = filtersElmt.getChildNodes();

        for (int i = 0; i < filtersChildren.getLength(); i++) {
            if (filtersChildren.item(i).getNodeName().equals("Filter")) {
                Node filter = filtersChildren.item(i);
                LogFilter logFilter = getLogFilterObj(filter);
                namedLogFilters.put(logFilter.getName(), logFilter);
            }
        }
        return namedLogFilters;
    }

    /*
     * Take a <Filter> node and return a LogFilter object
     */
    private LogFilter getLogFilterObj(Node filter) throws Exception {
        LogFilter logFilter = new LogFilter("", 0, LogBrowserConsts.DEFAULT_COLUMN_COUNT);
        NodeList filterChildren = filter.getChildNodes();
        for (int j = 0; j < filterChildren.getLength(); j++) {
            Node filterChild = filterChildren.item(j);
            if (filterChild.getNodeName().equals("Name")) {
                Node cdataName = filterChild.getFirstChild();  // Get CDATA node within <Name>
                if ((cdataName != null) && (cdataName.getNodeType() == Node.CDATA_SECTION_NODE)) {
                    logFilter.setName(cdataName.getTextContent());
                }
            } else if (filterChild.getNodeName().equals("FilterRows")) {
                Node filterRows = filterChild;
                populateLogFilterRows(logFilter, filterRows);
            } else if (filterChild.getNodeName().equals("NumSurroundingLines")) {
                Node numSurrLines = filterChild;
                logFilter.setNumSurroundingLines(Integer.parseInt(numSurrLines.getTextContent()));
            }
        }
        return logFilter;
    }

    /*
     * Take the <FilterRows> element and populate the LogFilter object row by row
     */
    private void populateLogFilterRows(LogFilter logFilter, Node filterRows) throws Exception {
        NodeList filterRowsChildren = filterRows.getChildNodes();
        int filterRowIndex = 0;     // Counter for actual <FilterRow> elements
        for (int frsChildIndex = 0; frsChildIndex < filterRowsChildren.getLength(); frsChildIndex++) {
            Node filterRowsChild = filterRowsChildren.item(frsChildIndex);
            if (filterRowsChild.getNodeName().equals("FilterRow")) {
                Node filterRow = filterRowsChild;
                NodeList filterRowChildren = filterRow.getChildNodes();
                for (int j = 0; j < filterRowChildren.getLength(); j++) {
                    if (filterRowChildren.item(j).getNodeName().equals("FilterItems")) {
                        Node filterItems = filterRowChildren.item(j);
                        NodeList filterItemsChildren = filterItems.getChildNodes();
                        int filterItemIndex = 0;    // Counter for actual <FilterItem> elements
                        for (int fisChildIdx = 0; fisChildIdx < filterItemsChildren.getLength(); fisChildIdx++) {
                            if (filterItemsChildren.item(fisChildIdx).getNodeName().equals("FilterItem")) {
                                Node filterItem = filterItemsChildren.item(fisChildIdx);
                                populateLogFilterItems(logFilter, filterRowIndex, filterItemIndex, filterItem);
                                filterItemIndex++;
                            }
                        }
                    }
                }
                filterRowIndex++;
            }
        }
    }

    /*
     * For a given <FilterItem> node in a row, populate the LogFilter object's filter item
     * at that row index / column index.
     */
    private void populateLogFilterItems(LogFilter logFilter,
                                        int rowIndex,
                                        int filterItemIndex,
                                        Node filterItem) throws Exception {
        NodeList filterItemChildren = filterItem.getChildNodes();
        FilterItem fi = new FilterItem();
        for (int n = 0; n < filterItemChildren.getLength(); n++) {
            Node filterDatum = filterItemChildren.item(n);
            if (filterDatum.getNodeName().equals("BooleanConnector")) {
                fi.setBoolConnector(filterDatum.getTextContent());
            }
            if (filterDatum.getNodeName().equals("NOT")) {
                fi.setNegateFlag(filterDatum.getTextContent().equals("True"));
            }
            if (filterDatum.getNodeName().equals("RegEx")) {
                fi.setRegexFlag(filterDatum.getTextContent().equals("True"));
            }
            if (filterDatum.getNodeName().equals("Keyword")) {
                Node cdataKeyword = filterDatum.getFirstChild();  // Get CDATA node within <Keyword>
                if ( (cdataKeyword != null) &&
                     (cdataKeyword.getNodeType() == Node.CDATA_SECTION_NODE)) {
                    fi.setKeyword(cdataKeyword.getTextContent());
                }
            }
        }
        logFilter.setFilterItem(rowIndex, filterItemIndex, fi);
    }

    public Document getConfigDoc() {
        return configDoc;
    }

}
