/***********************************************************
 * Copyright 2010 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core.parser;

import java.util.Date;

import com.vmware.vide.vlogbrowser.core.model.ISingleMatcher;
import com.vmware.vide.vlogbrowser.core.model.LogItem;

/**
 * The DateMatcher class is used to handle filtering of LogItems from the Log
 * Table view. It checks each cell in the LogItem object to see if it matches the filter
 * date within the specified tolerance. It is also used to find the closest date in a list
 * to the specified date.
 */
public class DateMatcher implements ISingleMatcher {

    private long filterDateMillis;
    private long tolerance;
    private int closestMatchDateIdx;
    private long filterDateClosestDateDiff;

    public DateMatcher(Date filterDate, Long tolerance) {
        this.filterDateMillis = filterDate.getTime();
        this.tolerance = tolerance;
        this.filterDateClosestDateDiff = Long.MAX_VALUE;
        this.closestMatchDateIdx = 0;
    }

    public DateMatcher(Date filterDate) {
        this(filterDate, new Long(0L));
    }

    @Override
    public boolean matches(LogItem logItem) {
        long diff = Math.abs(logItem.getLogDate().getTime() - filterDateMillis);
        if (diff <= tolerance) {
            return true;
        } else {
            return false;
        }
    }

    public void submitMatchCandidate(LogItem logItem, int listIndex) {
        long diff = Math.abs(logItem.getLogDate().getTime() - filterDateMillis);
        if (diff < filterDateClosestDateDiff) {
            filterDateClosestDateDiff = diff;
            closestMatchDateIdx = listIndex;
        }
    }

    public int getClosestMatchIndex() {
        return closestMatchDateIdx;
    }

}
