/***********************************************************
 * Copyright 2011 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/
package com.vmware.vide.vlogbrowser.core.parser;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;

/**
 * Implements the ILineReader for character encodings that use a 'LF' (i.e. 0x0A) as a end
 * of a line (EOL). The character encodings must not include a 0x0A in a character byte sequence
 * except a EOL.
 * The bytes sequence 0x0D 0x0A is handled as a EOL.
 */
public class LFLineReader implements ILineReader {
    private static final int EXPAND_DELTA = 1024;
    private static final int INITIAL_BUFFER_SIZE = 1024;

    private final BufferedInputStream bis;
    private final String charEncode;

    private byte[] bytebuffer = new byte[INITIAL_BUFFER_SIZE];
    /* Input pointer for the bytebuffer */
    private int inPtr = 0;

    LFLineReader(FileInputStream fis, String charEncode) {
        this.bis = new BufferedInputStream(fis);
        this.charEncode = charEncode;
    }

    @Override
    public int readLine(StringBuilder sb) throws IOException {
        int b;
        while ((b = bis.read()) != -1) {
            if (b == 0x0A) {
                int len = inPtr + 1;
                fillLineContents(sb);
                return len;
            }
            if (inPtr >= bytebuffer.length) {
                expandBuffer();
            }
            bytebuffer[inPtr] = (byte) b;
            ++inPtr;
        }
        if (inPtr > 0) {
            int len = inPtr;
            fillLineContents(sb);
            return len;
        }
        return -1;
    }

    @Override
    public void close() throws IOException {
        bis.close();
    }

    /**
     * Fill the line contents in the bytebuffer to the string builder.
     * This resets the input pointer for the bytebuffer.
     * @param sb The string builder.
     * @throws UnsupportedEncodingException This should not happen since Java promises UTF-8.
     */
    private void fillLineContents(StringBuilder sb) throws UnsupportedEncodingException {
        if (inPtr > 0 && bytebuffer[inPtr - 1] == 0x0D) {
            sb.append(new String(bytebuffer, 0, inPtr - 1, charEncode));
        } else {
            sb.append(new String(bytebuffer, 0, inPtr, charEncode));
        }
        inPtr = 0;
    }

    private void expandBuffer() {
        byte[] newarray = new byte[bytebuffer.length + EXPAND_DELTA];
        System.arraycopy(bytebuffer, 0, newarray, 0, bytebuffer.length);
        bytebuffer = newarray;
    }
}
