/***********************************************************
 * Copyright 2011 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/
package com.vmware.vide.vlogbrowser.core.parser;

import java.io.FileInputStream;
import java.io.IOException;
import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;

/**
 * Text line reader that reads lines from a file input stream with decoding
 * by the specified charset.
 */
public class LineReader implements ILineReader {
    private static final int REPLACEMENT_CHARACTER = 0xfffd;
    private static final int MAX_CHARENCODE_BYTES = 11;
    private final FileInputStream fis;
    private final Charset cset;
    /* The translation buffer */
    private final ByteBuffer bb = ByteBuffer.allocate(MAX_CHARENCODE_BYTES);

    private ByteBuffer buf = ByteBuffer.allocate(MAX_CHARENCODE_BYTES);

    private int readByte() throws IOException {
        if (buf.hasRemaining()) {
            return (int) buf.get();
        }
        return fis.read();
    }

    private void putBack(byte[] bytes) {
        ByteBuffer newBuf = ByteBuffer.allocate(MAX_CHARENCODE_BYTES);
        newBuf.put(bytes);
        newBuf.put(buf);
        buf = newBuf;
        buf.flip();
    }

    LineReader(FileInputStream fis, String charsetName) {
        this.fis = fis;
        cset = Charset.forName(charsetName);
        buf.flip();
    }

    /**
     * Read a character with decoding the buffered input stream by byte.
     *
     * Read a byte from the buffered input stream and add the byte to the translation buffer.
     * Decode the bytes in the translation buffer until the decoder successfully results a char.
     *
     * @param readIn returns read character.
     * @return number of bytes to read a character.
     * @throws IOException I/O error.
     */
    private int readChar(char[] readIn) throws IOException {
        int b;
        int count = 0;
        while ((b = readByte()) != -1) {
            ++count;
            bb.put((byte) b);
            bb.flip();
            byte[] ba = new byte[bb.limit()];
            bb.get(ba);
            ByteBuffer src = ByteBuffer.wrap(ba);
            CharBuffer dst = cset.decode(src);
            if (dst.limit() == 1 || dst.limit() == 2) {
                char ch = dst.charAt(dst.limit() - 1);
                if (ch == REPLACEMENT_CHARACTER) {
                    /*
                     * The bytes in the translation buffer are not long enough to be decoded.
                     * Read more byte to decode again.
                     */
                    if (bb.position() == bb.capacity()) {
                        /*
                         * Translation buffer full.
                         * Decode the first byte in the translation buffer as a replacement
                         * char, and prepare to restart decoding from the second byte in the
                         * next call.
                         */
                        bb.position(1);
                        byte[] remain = new byte[bb.remaining()];
                        bb.get(remain);
                        putBack(remain);
                        bb.position(0);
                        readIn[0] = REPLACEMENT_CHARACTER;
                        return 1;
                    }
                    /* Prepare the translation buffer to append a byte. */
                    bb.position(bb.limit());
                    bb.limit(bb.capacity());
                    continue;
                }
                bb.position(0);
                readIn[0] = ch;
                return count;
            } else {
                /*
                 * Here, the decoded pattern is one of
                 *  X O U
                 *  X O O
                 *  X X U
                 *  (where X is invalid char, O is decoded validly, U is not detected yet.)
                 * Return the second char and length of processed bytes including the first
                 * char. Since, we are not interested in the invalid char but in the processed
                 * bytes.
                 */
                char ch = dst.charAt(1);
                bb.position(0);
                byte[] b1 = { ba[ba.length - 1] };
                putBack(b1);
                readIn[0] = ch;
                return count - 1;
            }
        }
        if (bb.position() != 0) {
            /* eof happens in the middle of decoding. */
            readIn[0] = REPLACEMENT_CHARACTER;
            int ret = bb.position();
            bb.position(0);
            return ret;
        }
        return -1;
    }

    private static final int EMPTY = -2;
    private int prevChar = EMPTY;
    private int prevNum;

    private int readSingleChar(char[] readIn) throws IOException {
        if (prevChar != EMPTY) {
            readIn[0] = (char) prevChar;
            prevChar = EMPTY;
            return prevNum;
        }
        return readChar(readIn);
    }

    private void putBackChar(int num, char ch) {
        if (prevChar != EMPTY) {
            throw new BufferOverflowException();
        }
        prevNum = num;
        prevChar = (int) ch;
    }

    /**
     * Read a line and returns number of bytes of the line.
     * @param sb Appends characters in the read line.
     * @return number of bytes of the read line. -1 if reaches to the end of file.
     * @throws IOException I/O error
     */
    @Override
    public int readLine(StringBuilder sb) throws IOException {
        int num;
        int lineLen = 0;
        char[] readIn = new char[1];
        boolean eof = true;
        while ((num = readSingleChar(readIn)) != -1) {
            lineLen += num;
            eof = false;
            if (readIn[0] == '\n' || readIn[0] == '\r') {
                int extra = readChar(readIn);
                if (extra == -1) {
                    break;
                }
                if (readIn[0] == '\n' || readIn[0] == '\r') {
                    lineLen += extra;
                } else {
                    putBackChar(extra, readIn[0]);
                }
                break;
            }
            sb.append(readIn[0]);
        }
        if (eof) {
            return -1;
        }
        return lineLen;
    }

    /**
     * Close the stream.
     * @throws IOException I/O error.
     */
    @Override
    public void close() throws IOException {
        fis.close();
    }
}
