/***********************************************************

 * Copyright 2009 VMware, Inc.  All rights reserved.

 * -- VMware Confidential

 ***********************************************************/



package com.vmware.vide.vlogbrowser.core.parser;



import java.text.DateFormat;

import java.text.ParseException;

import java.text.SimpleDateFormat;

import java.util.Calendar;

import java.util.Date;

import java.util.GregorianCalendar;

import java.util.regex.Matcher;

import java.util.regex.Pattern;



import com.vmware.vide.vlogbrowser.core.model.LogDate;



/**

 * The LogDateFormat class takes a Date Pattern string of the kind used to create a

 * SimpleDateFormat object and uses it to attempt to parse strings of text containing dates,

 * returning LogDate objects. LogDateFormat also solves the "missing year" problem: many log

 * files are missing the year. Since the user is required to provide a date pattern

 * that is supported by Java's SimpleDateFormat class, if there are no 'y' characters present,

 * that means the date format is missing the year. In that case, LogDateFormat sets

 * yearMissing = true and allows each LogDate that it creates to use the default year: 1970.

 * Each LogDate is designed to call the LogDateFormat object that created it to determine the

 * correct timestamp to return to a caller. LogDateFormat figures out the correct adjustments

 * to make by taking the log file timestamp and subtracting an appropriate number of year

 * rollovers that were detected in the course of reading the entire log file. A year rollover

 * is assumed when December rolls to January, or February, etc. Please see the LogDate class

 * for more information.

 */

public class LogDateFormat {



    private DateFormat dateFormat;

    private int lastCalendarMonth;

    private Calendar currentCalendar;

    private boolean yearMissing;

    private Date logFilenameDate;

    private int logfileBeginningYear;

    private int numYearRollovers;

    private String datePattern;

    private long dateDelta;



    public LogDateFormat(String datePattern, Date fileTimestamp)

            throws IllegalArgumentException {

        this.dateFormat = new SimpleDateFormat(datePattern);

        this.datePattern = datePattern;

        this.dateDelta = 0L;

        this.yearMissing = true;

        boolean checkForY = true;

        for (int i = 0; i < datePattern.length(); i++) {

            if (datePattern.charAt(i) == '\'') {

                checkForY = !checkForY;

                continue;

            }

            if ((checkForY) && (datePattern.charAt(i) == 'y')) {

                this.yearMissing = false;

                break;

            }

        }

        initLogDate(fileTimestamp);

    }



    /** Creates a copy of date format to be used in multi-threaded parser */

    public LogDateFormat(LogDateFormat other, Date time, String filename) {

        this(other.datePattern, time);



        // try to get date from filename, to be used when log file entry has no date (only time)

        if (filename.length() > 0) {

            // try to parse year from filename

            Pattern pattern = Pattern.compile("(\\d\\d\\d\\d[_-]\\d\\d[_-]\\d\\d)");

            Matcher matcher = pattern.matcher(filename);

            if (matcher.find()) {

                String dateStr = matcher.group(1);

                String DATE_FORMATS[] = new String[] { "yyyy-MM-dd", "yyyy_MM_dd" };

                for (int i = 0; i < DATE_FORMATS.length; i++) {

                    SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_FORMATS[i]);

                    try {

                        logFilenameDate = dateFormat.parse(dateStr);

                        break;

                    }

                    catch (ParseException ex) {

                        // ignore - will try another date format

                    }

                }

            }

        }

    }



    public void initLogDate(Date fileTimestamp) {

        //this.logfileTimestamp = fileTimestamp;

        Calendar tempCalendar = new GregorianCalendar();

        tempCalendar.setTime(fileTimestamp);

        this.logfileBeginningYear = tempCalendar.get(Calendar.YEAR);



        this.lastCalendarMonth = -1;

        this.currentCalendar = new GregorianCalendar();

        this.numYearRollovers = 0;

    }



    public Date parse(String source) throws ParseException {

        // this call is not thread-reentrant, but different workers use different instances of dateFormat

        long currentDate = dateFormat.parse(source).getTime();



        // use date parsed from filename if not present in log entry

        if (yearMissing && logFilenameDate != null)

            currentDate += logFilenameDate.getTime();



        if ((yearMissing) && (yearRolledOver(currentDate))) {

            numYearRollovers++;

        }

        return new LogDate(currentDate, this, numYearRollovers);

    }



    private boolean yearRolledOver(long currentDate) {

        currentCalendar.setTimeInMillis(currentDate);

        boolean rolledOver = false;

        if ( (lastCalendarMonth >= 0) &&

             (lastCalendarMonth > currentCalendar.get(Calendar.MONTH))) {

            rolledOver = true;

            logfileBeginningYear--;

        }

        lastCalendarMonth = currentCalendar.get(Calendar.MONTH);

        return rolledOver;

    }



    public boolean isYearMissing() {

        return yearMissing;

    }



    public int getYear() {

        return logfileBeginningYear;

    }



    /**

     * Called by LogDate to correct its date (if yearMissing) before returning that value to its

     * caller.

     * @param dateInMillis

     * @param numRollovers

     * @return the corrected date as a long

     */

    public long correctYearlessDate(long dateInMillis, int numRollovers) {

        if (!yearMissing) {

            return dateInMillis;

        } else {

            // TODO: use pre-created instances of calendar

            Calendar calendar = new GregorianCalendar();

            calendar.setTimeInMillis(dateInMillis);

            calendar.set(Calendar.YEAR, logfileBeginningYear + numRollovers);

            return calendar.getTimeInMillis();

        }

    }



    public String getDatePattern() {

        return datePattern;

    }



    public void setDateDelta(long dateDelta) {

        this.dateDelta = dateDelta;

    }



    public long getDateDelta() {

        return dateDelta;

    }



}

