/***********************************************************

 * Copyright 2009 VMware, Inc.  All rights reserved.

 * -- VMware Confidential

 ***********************************************************/



package com.vmware.vide.vlogbrowser.core.parser;



import java.io.IOException;

import java.text.ParseException;

import java.util.ArrayList;

import java.util.Date;

import java.util.List;

import java.util.regex.Matcher;

import java.util.regex.Pattern;



import com.vmware.vide.vlogbrowser.core.model.ILogItemList;

import com.vmware.vide.vlogbrowser.core.model.LogDate;

import com.vmware.vide.vlogbrowser.core.model.LogItem;



/**

 * Created by the ConfigParser class, the LogFormat object provides a programmatic representation

 * of the XML configuration file used to describe a particular format of a log.

 * A log is assumed to a text file with multiple lines of text, where each line is a log

 * entry created at a particular time to describe some event. In a single log file of a certain

 * type, many sections of each log entry are common among all log entries, like the timestamp:

 * A timestamp is typically provided for each log entry and is always recorded in the same

 * location in the line, and using the same datetime pattern. Additionally, other sections of

 * each log entry, like the process that created the entry, appear in about the same location

 * and have the same notation line after line. (e.g. wrapped in [ ], separated by comma, colon,

 * semicolon, etc.)

 * Each of these sections of the log entry make up a field, and each field is assigned to a

 * column in a table. The columns can be used to sort log entries in the table.

 * The LogFormat class contains private inner classes that describe each log field and each

 * column to that will appear in the output table. Since it contains all the information needed

 * to parse and display each log entry, the LogFormat class also contains parsing rules and

 * methods, and is called by the LogParser class to do the actual parsing of a particular log

 * line. Even so, the LogFormat object operates essentially statelessly with respect to parsing

 * log lines; it is up to the caller of the LogFormat object to keep track of previously

 * parsed log entries, if necessary. The LogParser class calls LogFormat with a string of text

 * comprising a log line, and it also hands back to LogFormat the last valid LogItem that

 * LogFormat produced. The LogFormat class produces LogItem objects that contain a Date timestamp

 * and a sequence number, along with an offset and length. The offset and length are used to

 * parse text fields (strings) from the original source file after the LogItem was originally

 * created. In this way, memory is saved and load times are accelerated, since the parsing and

 * storage only has to be done to display the log item objects that currently appear on the

 * user's screen. Please see the LogItem class and ConfigParser class for more information.

 */

public class LogFormat {



    private final String name;

    private List<RAFile> randAccFiles;



    public static final int DEFAULT_LOWER_BOUND = 0;

    public static final int DEFAULT_UPPER_BOUND = Integer.MAX_VALUE;

    public static final int DEFAULT_REGEX_GROUP = 0;



    public static String newline = "\n";



    public enum ColumnType {

        LOGINDEX,

        TEXT,

        DATE

    }



    public static final String VALID_COL_TYPES = "LOGINDEX" +

                                                 " TEXT" +

                                                 " DATE";

    public static final ColumnType DEFAULT_COLUMN_TYPE = ColumnType.TEXT;



    public String getName() {

        return name;

    }



    private class Field {

        String name;

        //int index;

        int lowerBound;

        int upperBound;

        Pattern regexMatch;

        int regexCaptureGroup;



        Field (String name,

               int index,

               String lowerBound,

               String upperBound,

               String regexMatch,

               String regexCaptureGroup) throws Exception {



            if ((name == null) || (name.trim().equals(""))) {

                throw new Exception("Field name '" + name + "' is not valid");

            } else {

                this.name = name.trim();

            }



            //this.index = index;



            this.lowerBound = properNumberFromString(lowerBound,

                                                     "LowerBound",

                                                     DEFAULT_LOWER_BOUND,

                                                     DEFAULT_UPPER_BOUND,

                                                     DEFAULT_LOWER_BOUND);

            this.upperBound = properNumberFromString(upperBound,

                                                     "UpperBound",

                                                     this.lowerBound + 1,

                                                     DEFAULT_UPPER_BOUND,

                                                     DEFAULT_UPPER_BOUND);

            this.regexMatch = Pattern.compile(regexMatch.trim());

            this.regexCaptureGroup = properNumberFromString(regexCaptureGroup,

                                                            "RegexCaptureGroup",

                                                            DEFAULT_REGEX_GROUP,

                                                            DEFAULT_UPPER_BOUND,

                                                            DEFAULT_REGEX_GROUP);

        }

    }



    private final ArrayList<Field> fields;



    private Field getField(String name) {

        for (Field field : fields) {

            if (field.name.equals(name)) {

                return field;

            }

        }

        return null;

    }



    private int properNumberFromString(String numStr,

                                       String numStrName,

                                       int lowerLimit,

                                       int upperLimit,

                                       int defaultNum) throws Exception {

        int returnNum = 0;

        if ((numStr == null) || (numStr.trim().equals(""))) {

            returnNum = defaultNum;

        } else {

            try {

                returnNum = Integer.parseInt(numStr);

                if ((returnNum < lowerLimit) || (returnNum > upperLimit)) {

                    throw new Exception();

                }

            } catch (Exception e) {

                throw new Exception(numStrName + " '" + numStr +

                        "' is not valid. It must be left blank or be an integer between " +

                        lowerLimit + " and " + upperLimit + " inclusive.");

            }

        }



        return returnNum;

    }



    private class Column {

        String name;

        int index;

        boolean isDefault;

        int widthWeight;

        ArrayList<Field> fieldsAssigned;

        ColumnType columnType;

        private LogDateFormat columnDateFormat;



        String fieldGlue;



        Column (String name,

                int index,

                String isDefault,

                String widthWeight,

                String columnType,

                String dateFormat,

                String fieldGlue) throws Exception {



            if ((name == null) || (name.trim().equals(""))) {

                throw new Exception("Column name '" + name + "' is not valid. " +

                                    "It must have a valid non-empty name.");

            } else {

                this.name = name.trim();

            }



            this.index = index;



            if ((isDefault == null) ||

                (isDefault.equals("")) ||

                (isDefault.trim().equalsIgnoreCase("false"))) {

                this.isDefault = false;

            } else {

                this.isDefault = Boolean.parseBoolean(isDefault);

                if (this.isDefault != true) {

                    throw new Exception("IsDefault value '" + isDefault +

                                        "' is not valid. It must be left blank (defaults " +

                                        "to false), or be set to 'true' or 'false'");

                }

            }



            this.widthWeight = properNumberFromString(widthWeight,

                                                      "WidthWeight",

                                                      1,

                                                      DEFAULT_UPPER_BOUND,

                                                      1);



            if ((columnType == null) || (columnType.trim().equals(""))) {

                this.columnType = DEFAULT_COLUMN_TYPE;

            } else {

                try {

                    this.columnType = ColumnType.valueOf(columnType.trim());

                }

                catch (Exception e) {

                    throw new Exception("ColumnType '" + columnType +

                                        "' is not valid. It must be blank or have one " +

                                        "of these values: " + VALID_COL_TYPES);

                }

            }



            if ((dateFormat == null) || dateFormat.trim().equals("")) {

                this.columnDateFormat = null;

            } else {

                this.columnDateFormat = new LogDateFormat(dateFormat, new Date());

            }



            if (((this.columnType == ColumnType.DATE) && (this.columnDateFormat == null)) ||

                ((this.columnType != ColumnType.DATE) && (this.columnDateFormat != null))) {

                throw new Exception("Column Type " + columnType + " does not match" +

                        " DateFormat in column " + name + ".");

            }



            if ((this.columnType == ColumnType.LOGINDEX) && (this.isDefault == true)) {

                throw new Exception("The Column of type LOGINDEX cannot be the default column");

            }



            this.fieldsAssigned = new ArrayList<Field>(10);

            if (fieldGlue == null) {

                this.fieldGlue = "";

            } else {

                this.fieldGlue = fieldGlue;

            }



            if ((this.columnType == ColumnType.LOGINDEX) && (!this.fieldGlue.equals(""))) {

                throw new Exception("In the Column of type LOGINDEX, FieldGlue must be left unset.");

            }

        }



        public void updateDateFormatObj(Date newLogFileDate) throws Exception {

            try {

                this.columnDateFormat.initLogDate(newLogFileDate);

            } catch (IllegalArgumentException e) {

                throw new Exception("DateFormat '" + columnDateFormat + "' in column + " + name +

                        " is invalid. " + e.getMessage());

            }

        }



        public void setDateDelta(long dateDelta) throws Exception {

            this.columnDateFormat.setDateDelta(dateDelta);

        }



        public Long getDateDelta() {

            if (columnDateFormat == null) {

                return null;

            } else {

                return columnDateFormat.getDateDelta();

            }

        }



        public LogDateFormat getDateFormat() {

            return columnDateFormat;

        }

    }



    private final ArrayList<Column> columns;



    private Column defaultColumn;

    private Column dateColumn;

    private Column logIndexColumn;



    private int defaultColumnIndex = -1;

    public int getDefaultColumnIndex() {

        return defaultColumnIndex;

    }



    private int dateColumnIndex = -1;

    public int getDateColumnIndex() {

        return dateColumnIndex;

    }



    private int logIndexColumnIndex = -1;

    public int getLogIndexColumnIndex() {

        return logIndexColumnIndex;

    }



    private Column getColumn(String name) {

        for (Column column : columns) {

            if (column.name.equals(name)) {

                return column;

            }

        }

        return null;

    }



    public String[] getColumnNames() {

        String[] retStringArr = new String[columns.size()];

        for (int i = 0; i < columns.size(); i++) {

            retStringArr[i] = columns.get(i).name;

        }

        return retStringArr;

    }



    public int[] getColumnWeights() {

        int[] retIntArr = new int[columns.size()];

        for (int i = 0; i < columns.size(); i++) {

            retIntArr[i] = columns.get(i).widthWeight;

        }

        return retIntArr;

    }



    private int sizeLimitMB = -1;

    public int getSizeLimitMB() {

        return sizeLimitMB;

    }



    public void setSizeLimitMB(int value) {

        sizeLimitMB = value;

    }



    public LogFormat(String name) throws Exception {

        if ((name == null) || (name.trim().equals(""))) {

            throw new Exception("Log format must be named");

        }

        this.name = name.trim();



        fields = new ArrayList<Field>(10);

        columns = new ArrayList<Column>(10);

        knownFilenames = new ArrayList<String>(10);



    }



    private final List<String> knownFilenames;



    public void addKnownFilename(String regexFilename) {

        knownFilenames.add(regexFilename);

    }



    public List<String> getKnownFilenames() {

        return knownFilenames;

    }



    public void addField(String name,

                         String lowerBound,

                         String upperBound,

                         String regexMatch,

                         String regexCaptureGroup) throws Exception {



        try {

            if (getField(name) != null) {

                throw new Exception("Field name is not unique.");

            }

            fields.add(new Field(name,

                                 fields.size(),

                                 lowerBound,

                                 upperBound,

                                 regexMatch,

                                 regexCaptureGroup));

        } catch (Exception e) {

            throw new Exception("Bad Field '" + name + "' in LogFormat '" + this.name + "': " +

                                e.getMessage());

        }

    }



    public void addColumn(String name,

                          String isDefault,

                          String widthWeight,

                          String columnType,

                          String dateFormat,

                          String fieldGlue) throws Exception {



        try {

            if (getColumn(name) != null) {

                throw new Exception("Column name is not unique.");

            }

            Column column = new Column(name,

                                       columns.size(),

                                       isDefault,

                                       widthWeight,

                                       columnType,

                                       dateFormat,

                                       fieldGlue);

            columns.add(column);

            if (column.isDefault) {

                if (defaultColumn != null) {

                    throw new Exception("Only one column can be the default." +

                            " Duplicate found at column " + name);

                }

                defaultColumn = column;

                defaultColumnIndex = defaultColumn.index;

            }

            if (column.getDateFormat() != null) {

                if (dateColumn != null) {

                    throw new Exception("Only one column can be the DATE column." +

                            " Additional DATE found at column " + name);

                }

                dateColumn = column;

                dateColumnIndex = dateColumn.index;

            }

            if (column.columnType == ColumnType.LOGINDEX) {

                if (logIndexColumn != null) {

                    throw new Exception("Only one column can be the LOGINDEX column." +

                            " Additional LOGINDEX found at column " + name);

                }

                logIndexColumn = column;

                logIndexColumnIndex = logIndexColumn.index;

            }

            if ((column.isDefault) && (column.getDateFormat() != null)) {

                throw new Exception("Invalid type DATE found at default column " + name);

            }

            if ((column.isDefault) && (column.columnType == ColumnType.LOGINDEX)) {

                throw new Exception("The LOGINDEX column " + name + " cannot be the" +

                        " default column.");

            }

        } catch (Exception e) {

            throw new Exception("Bad Column '" + name + "' in LogFormat '" + this.name + "': " +

                                e.getMessage());

        }

    }



    public void addFieldToColumn(String fieldName, String columnName) throws Exception {



        if ((columnName == null) || (getColumn(columnName.trim()) == null)) {

            throw new Exception("Column '" + columnName + "' is invalid.");

        }



        if ((fieldName == null) || (getField(fieldName.trim()) == null)) {

            throw new Exception("Cannot find field '" +

                                fieldName +

                                "' referenced within column " + columnName +

                                " within LogFormat " + this.name);

        }



        Column columnObj = getColumn(columnName.trim());

        Field fieldObj = getField(fieldName.trim());



        if (columnObj.columnType == ColumnType.LOGINDEX) {

            throw new Exception("The LOGINDEX column " + columnName +

                    " cannot have any fields assigned to it.");

        }



        columnObj.fieldsAssigned.add(fieldObj);



    }



    private Column getColumnByAssignedField(String fieldName) {

        for (Column column : columns) {

            for (Field field : column.fieldsAssigned) {

                if (fieldName.equals(field.name)) {

                    return column;

                }

            }

        }

        return null;

    }



    public LogItem initialParse(String logLine,

            LogItem lastLogItem,

                                long fileOffset,

                                int readLength,

                                int logFileNum,

                                long logIndexNum, LogDateFormat dateFormat) {



        // try to find Date field

        Column myColumn = getColumnByAssignedField("Date");  // FIXME: Make this a true key field, defined in XML

        Object dateObj = getLogCell(myColumn.index, logLine, dateFormat);

        if (dateObj != null) {

            return new LogItem(true,

                               fileOffset,

                               readLength,

                               (Date) dateObj,

                               logIndexNum,

                               logFileNum);

        } else {

            if (lastLogItem != null) {

                lastLogItem.appendLogExtra(readLength);

                return null;

            } else {

                return new LogItem(false,

                                   fileOffset,

                                   readLength,

                                   new LogDate(0L, dateFormat, 0),

                                   logIndexNum,

                                   logFileNum);

            }

//            int lastLogItemIndex = logItemList.size() - 1;

////            int lastLogItemIndex = (int)logIndexNum - 1;

////            LogItem lastLogItem = (lastLogItemIndex > 0) ? logItemList.get(lastLogItemIndex, true) : null;

//            LogItem lastLogItem = logItemList.get(lastLogItemIndex) ;

//            if (lastLogItem  != null) { // I need to append the extra info into the last logitem of the list

//                lastLogItem.appendLogExtra(readLength);

//                logItemList.updateLogItem(lastLogItemIndex, lastLogItem);

//                return null;

//            } else {

//                return new LogItem(false,

//                                   fileOffset,

//                                   readLength,

//                                   new LogDate(0L, dateFormat, 0),

//                                   logIndexNum,

//                                   logFileNum);

//            }

        }

    }



    /**

     * Returns a table cell contents for log entry

     * Convenience method to keep old API, delegates to the next version of getLogCell()

     */

    public Object getLogCell(int columnIndex, String logLine) {

        LogDateFormat dateFormat = getDateFormat();

        return getLogCell(columnIndex, logLine, dateFormat);

    }



    /**

     * Returns a table cell contents for log entry

     *

     * @param columnIndex log column

     * @param logLine     log entry index

     * @param dateFormat  date format used to represent date/time

     * @return            cell contents text or object

     */

    public Object getLogCell(int columnIndex, String logLine, LogDateFormat dateFormat) {

        String retStr = "";

        Column myColumn = columns.get(columnIndex);

        List<Field> columnFields = myColumn.fieldsAssigned;

        for (Field field : columnFields) {

            String fieldStr = "";

            int lowerBound = field.lowerBound;

            int upperBound =

                (field.upperBound > logLine.length()) ? logLine.length() : field.upperBound;

            boolean found = false;

            if ((lowerBound < logLine.length()) && (lowerBound < upperBound)) {

                fieldStr = logLine.substring(lowerBound, upperBound);

                if ((field.regexMatch != null) && (field.regexMatch.toString().length() > 0)) {

                    Matcher matcher = field.regexMatch.matcher(fieldStr);

                    if (matcher.find()) {

                        try {

                            fieldStr =

                                fieldStr.substring(matcher.start(field.regexCaptureGroup),

                                                   matcher.end(field.regexCaptureGroup));

                            found = true;

                        } catch (IndexOutOfBoundsException e) {

                            found = false;

                        }

                    }

                } else {

                    found = true;

                }

            }

            if (myColumn.columnType == ColumnType.DATE) {

                if (found) {

                    try {

                        // PR# 767777: here dateFormat is a copy of original

                        // dateFormat, and it is used here only to parse date

                        // in LogDate we store original dataFormat used by the

                        // column

                        LogDate result = (LogDate) dateFormat.parse(fieldStr);

                        result.setLogDateFormat(myColumn.getDateFormat());

                        return result;

                    } catch (ParseException e) {

                        // This is normal; just fall through to the next statement

                    }

                }

                return null;

            } else if (found) {     // Fields located, destined for a non-Date column

                if (retStr.length() > 0) {

                    retStr += myColumn.fieldGlue;

                }

                retStr += fieldStr;

            }

        }

        return retStr;

    }



    public String getLogExtra(String logLine) {

        int nlIdx = logLine.indexOf(newline);

        if (nlIdx < 0) {

            return "";

        } else {

            return getLogSubstring(logLine, nlIdx + newline.length(), logLine.length());

        }

    }



    public String getLogPreamble(String logLine) {

        return getLogSubstring(logLine, 0, logLine.indexOf(newline));

    }



    private String getLogSubstring(String logLine, int begin, int end) {

        try {

            return logLine.substring(begin, end);

        } catch (IndexOutOfBoundsException e) {

            return "";

        }

    }



    public void loadNewRAFileList(List<RAFile> raFiles) throws IOException {

        closeAllRandAccFiles();

        if (randAccFiles != null) {

            randAccFiles.clear();

        }

        randAccFiles = raFiles;

    }



    public List<RAFile> getRandAccFileList() {

        return randAccFiles;

    }



    public RAFile getRandAccFile(int logFileNum) {

        return randAccFiles.get(logFileNum);

    }



    public void openRandAccFile(int logFileNum) throws IOException {

        randAccFiles.get(logFileNum).open();

    }



    public void openAllRandAccFiles() throws IOException {

        for (int i = 0; i < randAccFiles.size(); i++) {

            openRandAccFile(i);

        }

    }



    public void closeRandAccFile(int logFileNum) throws IOException {

        if ((randAccFiles != null) &&

            (randAccFiles.size() > logFileNum) &&

            (randAccFiles.get(logFileNum) != null)) {

            randAccFiles.get(logFileNum).close();

        }

    }



    public void closeAllRandAccFiles() throws IOException {

        if (randAccFiles != null) {

            for (int i = 0; i < randAccFiles.size(); i++) {

                closeRandAccFile(i);

            }

        }

    }



    public boolean matchesCompletely(String logLine) {

        for (Column column : columns) {

            if (column.columnType != ColumnType.LOGINDEX) {

                Object obj = getLogCell(column.index, logLine);

                if (obj == null) {

                    return false;

                } else if ((obj instanceof String) && (((String)obj).trim().length() == 0)) {

                    return false;

                }

            }

        }

        return true;

    }



    public void updateDate(Date fileTimestamp) throws Exception {

        dateColumn.updateDateFormatObj(fileTimestamp);

    }



    public void setDateDelta(long dateDelta) throws Exception {

        dateColumn.setDateDelta(dateDelta);

    }



    public Long getDateDelta() {

        return dateColumn.getDateDelta();

    }



    public LogDateFormat getDateFormat() {

        Column column = getColumnByAssignedField("Date");

        LogDateFormat result = column == null ? null : column.getDateFormat();

        return result;

    }



    public String getCharEncode() {

        /* TODO : need to support other encodings */

        return "UTF-8";

    }

}

