/***********************************************************
 * Copyright 2009 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core.parser;

import java.io.IOException;

import com.vmware.vide.vlogbrowser.core.model.LogFilter;
import com.vmware.vide.vlogbrowser.core.model.LogItem;

/**
 * The LogItemMatcher class is used to handle filtering of LogItems from the
 * Log Table View. It checks each cell in the LogItem object to see if it matches the filter
 * text. In some cases, the match checking is short-circuited. For example, the user may use
 * the "Include # Surrounding Lines" drop-down box. In this case, appropriate LogItems will be
 * set to "Always Match" by the updateFilter() method in the LogTableViewer class.
 *
 * This class, along with ComplexMatcher, both implement MultiMatcher which extends the
 * Matcher interface. LogTableViewer will use this class as a single keyword
 * matcher, and the ComplexMatcher class for more complex filtering operations.
 */
public class LogItemMatcher implements MultiMatcher {

    private String filterText;
    private LogFormat logFormat;
    private boolean isCaseSensitive;
    private boolean isJustAKeyword;

    public LogItemMatcher(String filterText, LogFormat logFormat) {
        this.filterText = filterText;
        this.logFormat = logFormat;
    }

    public LogItemMatcher(LogFilter logFilter, LogFormat logFormat) throws Exception {
        this(logFilter.getFirstKeyword(), logFormat);
        isCaseSensitive = logFilter.isCaseSensitive();
        isJustAKeyword = logFilter.isJustAKeyword();
    }

    @Override
    public boolean matches(LogItem logItem) {
        if (logItem.isAlwaysMatched()) {
            return true;
        }
        return matchesSomeField(logItem);
    }

    @Override
    public boolean matchesSomeField(LogItem logItem) {
//        try {
//            String logEntryText = new String(logItem.getLogEntryBytes(logFormat));
//            if (isCaseSensitive && isJustAKeyword) {
//                return (logEntryText.indexOf(filterText)) > -1;
//            } else { // it is _not_ case sensitive
//                return logEntryText.toLowerCase().indexOf(filterText.toLowerCase()) > -1;
//            }
//        } catch (IOException ioe) {
//            ioe.printStackTrace();
//        }
        boolean result = false;
        try {
            String logEntryText = new String(
                    logItem.getLogEntryBytes(logFormat));
            if (isCaseSensitive && isJustAKeyword) {
                result =  (logEntryText.indexOf(filterText)) > -1;
            } else { // it is _not_ case sensitive
                result = logEntryText.toLowerCase().indexOf(
                        filterText.toLowerCase()) > -1;
            }
//            if (result == true) {
//                System.out.println("["+logItem.getLogIndexNum()+"] "+logEntryText);
//            }
        } catch (IOException ioe) {
            ioe.printStackTrace();
        }
        return result;
    }

}
