/***********************************************************

 * Copyright 2009-2011 VMware, Inc.  All rights reserved.

 * -- VMware Confidential

 ***********************************************************/



package com.vmware.vide.vlogbrowser.core.parser;



import java.io.File;

import java.io.FileInputStream;

import java.io.IOException;

import java.util.Date;

import java.util.List;



import org.slf4j.Logger;

import org.slf4j.LoggerFactory;



import com.vmware.vide.vlogbrowser.core.fileops.BaseLogWorker;

import com.vmware.vide.vlogbrowser.core.model.FilterList;

import com.vmware.vide.vlogbrowser.core.model.LogItem;



/**

 * The LogParser class is used by a LogWorker job to do the work (the looping) necessary to

 * read in a log file and produce a List of LogItem objects. Each LogItem object is produced

 * by passing the offset and length of the log entry in the original log file. Additionally,

 * the loop also passes in the previously LogItem object, in case subsequent lines need to be

 * appended to it.

 *

 * Individual log files are parsed by separate workers in parallel, and then results are merged

 * together and loaded into table viewer.

 */

public class LogParser {



    private int logItemIndex = 0;

    

    private static final Logger logger = LoggerFactory.getLogger(LogParser.class);



    /**

     * Reads a log file and returns a list of log entries. Several instances of this method

     * may be called at the same time from different worker jobs

     * @param logFormat         log format instance used for parsing data

     * @param logFileNumber     index of the current log file

     * @param targetFilePath    log file path

     * @param raFile            remote file info

     * @param monitor           progress monitor

     * @return                  a list of parsed log entries

     * @throws IOException      exception if file cannot be read

     */

    private void readLogFile(LogFormat logFormat, int logFileNumber, String targetFilePath,

        RAFile raFile, FilterList filterList)

        throws IOException 

    {

        String charEncode = logFormat.getCharEncode();

        // setup monitor

        File targetFile = new File(targetFilePath);

        ILineReader reader = LineReaderFactory.createLineReader(new FileInputStream(targetFile), charEncode);



        // create a copy of date format for re-entrant parsing

        Date srcFileTimestamp = raFile.getSrcFileTimestamp();

        LogDateFormat format = logFormat.getDateFormat();

        LogDateFormat dateFormat = new LogDateFormat(format, srcFileTimestamp, raFile.getTargetFileName());



        long fileSize = targetFile.length();

        logger.debug("parsing " + targetFile.getName() + " (" + fileSize +" bytes)");



        LogItem lastLogItem = null;

        long fileOffset = 0L;

        int logItemLinePos = 0;



        StringBuilder lineBuf = new StringBuilder();

        int lineLength;

        while ((lineLength = reader.readLine(lineBuf)) != -1) {

            LogItem logItem = logFormat.initialParse(lineBuf.toString().trim(), lastLogItem,

                    fileOffset, lineLength, logFileNumber, logItemIndex, dateFormat);

//            logItemIndex++;

            fileOffset += lineLength;

            logItemLinePos++;

            lineBuf.setLength(0);



            if (logItem != null) {

                logItemIndex++;

                filterList.add(logItem);

                lastLogItem = logItem;

            } else if (lastLogItem != null) {

                

                filterList.updateLogItem(lastLogItem.getLogIndexNum(), lastLogItem);

            }

        }



        reader.close();

    }



    /**

     * Runs several log parsing jobs in parallel. This allows parser for each individual log

     * files to run in parallel, and then all parsed entries are loaded into

     */

    public void loadFromFiles(final LogFormat logFormat, final BaseLogWorker logWorker)

        throws Exception

    {

        // get list of files to process

        List<RAFile> raFiles = logFormat.getRandAccFileList();

        int nFiles = raFiles.size();

        if ((raFiles == null) || (nFiles == 0)) {

            return;

        }

        RAFile lastRaFile = logFormat.getRandAccFile(nFiles - 1);

        logFormat.updateDate(lastRaFile.getSrcFileTimestamp());

        

//        int nCpus = Runtime.getRuntime().availableProcessors();

//        ExecutorService pool = Executors.newFixedThreadPool(nCpus);

        

        // Reset the RA files in the case of File system list.

        // (see com.vmware.vide.vlogbrowser.core.model.LogItemListRAFile.clear)

        logWorker.getFilterList().clear();



        for (int currentFile = 0; (currentFile < nFiles); currentFile++) {

            final int logFileNumber = currentFile;



            // get next file to process

            final RAFile raFile = logFormat.getRandAccFile(currentFile);



            // wait until files is copied to local filesystem

            final String targetFilePath = raFile.getTargetFilePath(true);



            // open file after getTargetFilePath(true), to allow data access

            // from table label provider

            raFile.open();

            Runnable task = new Runnable(){

                @Override

                public void run() {

                    try {

//                      ILogItemList logItems = logWorker.getAllItems();

                      readLogFile(logFormat, logFileNumber, targetFilePath, raFile, logWorker.getFilterList());

                      logger.debug("adding log file " + logFileNumber);

                  } catch (Exception ex) {

                      logger.error("Error adding file " + logFileNumber, ex);

                  }                    

                }

            };

//            pool.submit(task);

            task.run();

        }

//        pool.shutdown();

//        pool.awaitTermination(60, TimeUnit.SECONDS);

    }

}



