/***********************************************************
 * Copyright 2010 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core.parser;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.vmware.vide.vlogbrowser.core.consts.LogBrowserConsts;

/**
 * The ManifestFile class encapsulates a manifest file stored in the staging directory used by
 * a particular vLogBrowser configuration file (.logx file). The manifest file stores the list
 * of log files that were cached in that staging directory by vLogBrowser. The next time that
 * vLogBrowser file (.logx file) is opened, the manifest file is located (if available). The
 * manifest file is then checked, and if it still points to a list of valid, cached log files,
 * those log files are loaded into the vLogBrowser log viewer table. If the manifest file is
 * somehow invalid, or a reload is forced, a new set of log files is downloaded/copied from the
 * source and the manifest file is rewritten to store information about those files.
 */
public class ManifestFile {

    private File mfstFile;
    private List<RAFile> raFiles;
    private Document xmlDocument;

    public static final String MANIFEST_TEMPLATE = "manifest.xml";

    public ManifestFile(String parentDir,
                        String configFilename,
                        boolean overwriteManifest) throws Exception {
        this.mfstFile = new File(parentDir, MANIFEST_TEMPLATE);
        if ((overwriteManifest) || !(isLegalManifest(mfstFile))) {
            initNewManifestFile(configFilename);
        } else {
            this.xmlDocument = XmlDocUtils.parseXmlFromFile(mfstFile);
        }
        this.raFiles = new ArrayList<RAFile>();
        loadFileList();
    }

    public void initNewManifestFile(String configFilename) throws Exception {
        mfstFile.delete();
        mfstFile.createNewFile();
        createNewManifest(configFilename);
    }

    public static boolean isLegalManifest(File testFile) {
        try {
            Document manifestDoc = XmlDocUtils.parseXmlFromFile(testFile);
            String configFileValue = XmlDocUtils.getElement(manifestDoc, "/root/ConfigFile");

            if ((configFileValue == null) || (configFileValue.trim().length() == 0)) {
                return false;
            }

            // The StagedLogFiles element which is the parent of the File elements
            XmlDocUtils.getElement(manifestDoc, "/root/StagedLogFiles");
        } catch (Exception e) {
            return false;
        }
        return true;
    }

    public boolean isValidFileList() {
        for (RAFile raFile : raFiles) {
            try {
                raFile.open();
                raFile.close();
            } catch (Exception e) {
                return false;
            }
        }
        return true;
    }

    public void createNewManifest(String configFilename) throws Exception {
        InputStream templateFile = getManifestTemplate();
        if (templateFile != null) {
            xmlDocument = XmlDocUtils.parseXmlFromInputStream(templateFile);
            XmlDocUtils.replaceXmlElement(xmlDocument, "ConfigFile",
                    configFilename);
            flushToDisk();
        } else {
            throw new Exception("Unable to find the manifest template file!");
        }
    }

    public void flushToDisk() throws IOException {
        String xmlContents = XmlDocUtils.getXmlStringFromDocument(xmlDocument);
        BufferedWriter bufferedOut = new BufferedWriter(new FileWriter(mfstFile.getCanonicalFile()));
        bufferedOut.write(xmlContents);
        bufferedOut.close();
    }

    public void loadFileList() {
        try {
            String[] fileIDArr = null;
            fileIDArr = XmlDocUtils.getElements(xmlDocument,
                                                "/root/StagedLogFiles/File",
                                                "SeqNum");

            if ((fileIDArr == null) || (fileIDArr.length == 0)) {
                return;
            }
            DateFormat stdDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm");

            for (int i = 0; i < fileIDArr.length; i++) {
                String pathPrefix = "/root/StagedLogFiles/File[SeqNum='" + fileIDArr[i] + "']/";
                String srcHostname = XmlDocUtils.getElement(xmlDocument,
                                                            pathPrefix + "SrcHostname[1]");
                String srcFilePath = XmlDocUtils.getElement(xmlDocument,
                                                            pathPrefix + "SrcFilePath[1]");
                String targetFilePath = XmlDocUtils.getElement(xmlDocument,
                                                               pathPrefix + "TargetFilePath[1]");
                String date = XmlDocUtils.getElement(xmlDocument, pathPrefix + "Date[1]");
                String size = XmlDocUtils.getElement(xmlDocument, pathPrefix + "Size[1]");
                RAFile raFile = new RAFile();
                raFile.setSrcFileHost(srcHostname);
                raFile.setSrcFilePath(srcFilePath);
                raFile.setSrcFileTimestamp(stdDateFormat.parse(date));
                raFile.setSizeInBytes(Integer.parseInt(size));
                raFile.setTargetFilePath(targetFilePath);
                raFiles.add(raFile);
            }

        } catch (Exception e) {
            return;
        }

    }


    public void addManifestFileNode(String seqNum,
                                    String srcHostname,
                                    String srcFilePath,
                                    String targetFilePath,
                                    String date,
                                    String size) throws Exception {

        Element seqNumElmt = xmlDocument.createElement("SeqNum");
        seqNumElmt.appendChild(xmlDocument.createTextNode(seqNum));

        Element hostnameElmt = xmlDocument.createElement("SrcHostname");
        hostnameElmt.appendChild(xmlDocument.createTextNode(srcHostname));

        Element srcFilePathElmt = xmlDocument.createElement("SrcFilePath");
        srcFilePathElmt.appendChild(xmlDocument.createTextNode(srcFilePath));

        Element tgtFilePathElmt = xmlDocument.createElement("TargetFilePath");
        tgtFilePathElmt.appendChild(xmlDocument.createTextNode(targetFilePath));

        Element dateElmt = xmlDocument.createElement("Date");
        dateElmt.appendChild(xmlDocument.createTextNode(date));

        Element sizeElmt = xmlDocument.createElement("Size");
        sizeElmt.appendChild(xmlDocument.createTextNode(size));

        Element fileElmt = xmlDocument.createElement("File");
        fileElmt.appendChild(seqNumElmt);
        fileElmt.appendChild(hostnameElmt);
        fileElmt.appendChild(srcFilePathElmt);
        fileElmt.appendChild(tgtFilePathElmt);
        fileElmt.appendChild(dateElmt);
        fileElmt.appendChild(sizeElmt);

        Node rootNode = xmlDocument.getFirstChild();

        Element stagedLFElement = null;
        NodeList nodeList = rootNode.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            if (nodeList.item(i).getNodeName().equals("StagedLogFiles")) {
                stagedLFElement = (Element) nodeList.item(i);
                break;
            }
        }
        if (stagedLFElement == null) {
            stagedLFElement = xmlDocument.createElement("StagedLogFiles");
            rootNode.appendChild(stagedLFElement);
        }

        stagedLFElement.appendChild(fileElmt);
    }

    public void addRAFiles(List<RAFile> moreRAFiles) throws Exception {
        if ((moreRAFiles == null) || (moreRAFiles.size() == 0)) {
            return;
        }
        DateFormat stdDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm");
        for (RAFile raFile : moreRAFiles) {
            String tfp = raFile.getTargetFilePath(false);
            if (tfp == null) {
                tfp = "";
            }
            addManifestFileNode(Integer.toString(raFiles.size()),
                                raFile.getSrcFileHost(),
                                raFile.getSrcFilePath(),
                                tfp,
                                stdDateFormat.format(raFile.getSrcFileTimestamp()),
                                Integer.toString(raFile.getSizeInBytes()));
            raFiles.add(raFile);
        }
        flushToDisk();
    }

    public Document getDocument() {
        return xmlDocument;
    }

    public List<RAFile> getFileList() {
        return raFiles;
    }

    public File getFile() {
        return mfstFile;
    }

    public InputStream getManifestTemplate() {
        return getClass().getResourceAsStream(LogBrowserConsts.TEMPLATE_DIR + MANIFEST_TEMPLATE);
    }

}
