/***********************************************************

 * Copyright 2009 VMware, Inc.  All rights reserved.

 * -- VMware Confidential

 ***********************************************************/



package com.vmware.vide.vlogbrowser.core.parser;



import java.io.File;

import java.io.IOException;

import java.io.RandomAccessFile;

import java.text.DateFormat;

import java.text.SimpleDateFormat;

import java.util.Date;



import com.vmware.vide.vlogbrowser.core.fileops.LogFileUtils;



/**

 * The RAFile class is used to encapsulate a file object that will be randomly accessed.

 * It contains methods provided for convenient access to the file, for sequential reading

 * (delimited by newlines) as well as reading a line at a particular offset and length.

 * The LogFormat class maintains a reference to an RAFile object from which it fulfills read

 * requests for fields in LogItem objects.

 */

public class RAFile implements Comparable<RAFile> {



    private String targetFilePath;

    private File file;

    private RandomAccessFile rafile;

    private String nextLine;



    // The directory location and file name of the log file (may be inside a vm-support .tgz file)

    private String srcFilePath;

    private String srcFileHost;

    private char srcSeparator;



    // The directory location and file name of the archive (.tgz) that stores the log file,

    // or null if not stored in an archive

    private String srcArchivePath;



    private Date srcFileTimestamp;

    private int sizeInBytes;



    private boolean isArchive;

    private boolean seqNumInvertedForSort;



    /** Indicates that copying and parsing of this file was canceled */

    private boolean isCanceled;



    public RAFile() {

        seqNumInvertedForSort = false;

    }



    public RAFile(String targetFilePath) throws Exception {

        this();

        this.targetFilePath = targetFilePath;

        this.file = new File(this.targetFilePath);

        this.rafile = new RandomAccessFile(file, "r");

    }



    public String getNextLine() throws IOException {

        return nextLine;

    }



    public void resetFilePointer() throws IOException {

        rafile.seek(0);

    }



    public int loadNextLine() throws IOException {

        long initialPosition = rafile.getFilePointer();

        nextLine = rafile.readLine();

        if (nextLine != null) {

            return new Long(rafile.getFilePointer() - initialPosition).intValue();

        } else {

            return 0;

        }

    }



    public int readBytes(byte[] buff, long offset) throws IOException {

        rafile.seek(offset);

        return rafile.read(buff, 0, buff.length);

    }



    public int readChars(char[] buff, long offset) throws IOException {

        byte[] myByteBuff = new byte[buff.length];

        int bytesRead = readBytes(myByteBuff, offset);

        for (int i = 0; i < buff.length; i++) {

            buff[i] = (char)myByteBuff[i];

        }

        return bytesRead;

    }



    public long getCurrentPointer() throws IOException {

        return rafile.getFilePointer();

    }



    public void close() throws IOException {

        if (rafile != null) {

            rafile.close();

        }

    }



    /** Sets flag that file loading was canceled, it notifies all jobs waiting for this file */

    public synchronized void setCanceled(boolean isCanceled) {

        this.isCanceled = isCanceled;

        notifyAll();

    }



    /** Sets target file path after file was copied, and notifies all jobs waiting for this file */

    public synchronized void setTargetFilePath(String targetFilePath) {

        this.targetFilePath = targetFilePath;

        notifyAll();

    }



    public synchronized String getTargetFilePath(boolean waitUntilReady) {

        if (!waitUntilReady && (targetFilePath == null)) {

            return null;

        }



        if (targetFilePath == null && !isCanceled) {

            try {

                wait();

            }

            catch (InterruptedException e) {

                e.printStackTrace();

            }

        }

        return targetFilePath;

    }



    public String getTargetFileName() {

        return LogFileUtils.getFileName(getTargetFilePath(true));

    }



    public void setSrcFilePath(String srcFilePath) {

        this.srcFilePath = srcFilePath;

    }



    public String getSrcFilePath() {

        return srcFilePath;

    }



    public String getSrcFileName(boolean trimFileExtension) {

        if (trimFileExtension) {

            return LogFileUtils.getFileNameWithoutExt(srcFilePath);

        } else {

            int index = srcFilePath.lastIndexOf(srcSeparator);

            return srcFilePath.substring(index + 1);

        }

    }



    public void setSrcFileHost(String srcFileHost) {

        this.srcFileHost = srcFileHost;

    }



    public String getSrcFileHost() {

        return srcFileHost;

    }



    public void setSrcFileTimestamp(Date srcFileTimestamp) {

        this.srcFileTimestamp = srcFileTimestamp;

    }



    public Date getSrcFileTimestamp() {

        return srcFileTimestamp;

    }



    public void setSizeInBytes(int size) {

        sizeInBytes = size;

    }



    public int getSizeInBytes() {

        return sizeInBytes;

    }



    public void setSeqNumInvertedForSort(boolean seqNumInvertedForSort) {

        this.seqNumInvertedForSort = seqNumInvertedForSort;

    }



    public boolean isSeqNumInvertedForSort() {

        return seqNumInvertedForSort;

    }



    public void setSrcArchivePath(String srcArchivePath) {

        this.srcArchivePath = srcArchivePath;

    }



    public String getSrcArchivePath() {

        return srcArchivePath;

    }



    @Override

    public int compareTo(RAFile otherRAFile) {

        int compareResult = getFileSeqNum().compareTo(otherRAFile.getFileSeqNum());

        if (seqNumInvertedForSort) {

            compareResult = -compareResult;

        }

        if ((compareResult == 0) ||

                (! getSrcFileName(true).equals(otherRAFile.getSrcFileName(true)))) {

            compareResult = srcFileTimestamp.compareTo(otherRAFile.getSrcFileTimestamp());

            if (compareResult == 0) {

                compareResult = getSrcFileName(false).compareTo(otherRAFile.getSrcFileName(false));

            }

        }

        return compareResult;

    }



    public void setArchive(boolean isArchive) {

        this.isArchive = isArchive;

    }



    public boolean isArchive() {

        return isArchive;

    }



    @Override

    public String toString() {

        DateFormat stdDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm");

        String dateStr = stdDateFormat.format(srcFileTimestamp);

        return srcFileHost + "|" + srcFilePath + "|" + targetFilePath + "|" + dateStr +

            "|" + sizeInBytes;

    }



    public String toDisplayString() {

        DateFormat stdDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm");

        String dateStr = stdDateFormat.format(srcFileTimestamp);

        return srcFileHost + "; " + srcFilePath + "  " + dateStr + "  " + sizeInBytes;

    }



    public String toShortDisplayString() {

        return srcFileHost + "; " + srcFilePath;

    }



    public void open() throws IOException {

        if (rafile != null) {

            rafile.close();

        }

        file = new File(targetFilePath);

        rafile = new RandomAccessFile(file, "r");

    }



    public Integer getFileSeqNum() {

        String fileName = getSrcFileName(false);

        String seqNumStr = LogFileUtils.getLastSubStringByRegEx(fileName, "\\d+", 0);

        Integer seqNum = new Integer(Integer.MIN_VALUE);

        try {

            seqNum = Integer.parseInt(seqNumStr);

        } catch (Exception e) {

            // If no File Sequence Number available, give up and default to minimum integer

        }

        return seqNum;

    }



    public void setSrcSeparator(char c) {

        srcSeparator = c;

    }



}

