/***********************************************************

 * Copyright 2012 VMware, Inc.  All rights reserved.

 * -- VMware Confidential

 ***********************************************************/



package com.vmware.vide.vlogbrowser.core.parser;



import java.io.DataInputStream;

import java.io.EOFException;

import java.io.File;

import java.io.FileInputStream;

import java.io.FileWriter;

import java.io.IOException;



import org.slf4j.Logger;

import org.slf4j.LoggerFactory;



import flex.messaging.io.SerializationContext;

import flex.messaging.io.amf.AbstractAmfInput;

import flex.messaging.io.amf.Amf0Input;

import flex.messaging.io.amf.Amf3Input;



/**

 * This class is responsible to parse the Local Shared Object file (.SOL)

 * originated by the Flex client framework. All data in the SOL file format is

 * AMF-based so it required us to reverse engineer the file and cross reference

 * it with the following document:

 * http://opensource.adobe.com/wiki/download/attachments/1114283/amf3_spec_05_05_08.pdf

 *

 * @author gcaprino

 *

 */

public class SolFileParser {



    private static final Logger logger = LoggerFactory.getLogger(SolFileParser.class);



    /**

     * This method deserialize SOL messages<code>input</code>, and write the

     * output in a textfile <code>output</code>

     *

     * @param inputFile

     *            SOL file

     * @param outputFile

     *            TXT file

     */

    public static void decode(File inputFile, File outputFile) throws IOException, ClassNotFoundException {

        DataInputStream dataInputStream = new DataInputStream(new FileInputStream(inputFile));

        byte amfVersion = skipHeader(dataInputStream);

        SerializationContext context = new SerializationContext();

        AbstractAmfInput parser = amfVersion == 3 ? new Amf3Input(context) : new Amf0Input(

                context);

        parser.setInputStream(dataInputStream);

        

        outputFile.createNewFile();

        FileWriter writer = new FileWriter(outputFile);

        

        try {

            boolean done = false;

            while (!done) { // !f.eof()

                try {

                    byte enc = dataInputStream.readByte();

                    if ((enc & 1) == 0) {

                        throw new IOException("SOL Format not supported at: "

                                + inputFile.getAbsolutePath());

                    }

                    byte[] nam = new byte[enc >> 1];

                    dataInputStream.read(nam);

                    byte typ = dataInputStream.readByte();

                    switch (typ) {

                    case 9: // array

                        int count = readU29AValue(dataInputStream);

                        typ = dataInputStream.readByte();

                        if (typ != 1) {

                            throw new IOException(

                                    "SOL Format not supported at: "

                                            + inputFile.getAbsolutePath());

                        }

                        int i;

                        for (i = 0; i < count; ++i) {

                            Object object = null;

                            try {

                                object = parser.readObject();

                            } catch (IndexOutOfBoundsException e) {

                                // do nothing, keep reading

                            }

                            if (object != null) {

                                String entry = object.toString();

                                if (!entry.isEmpty()) {

                                    entry = cleanEntry(entry);

                                    writer.write(entry);

                                }

                            }

                        }

                        continue;

                    default:

                        break;

                    }

                    parser.readObject(); // read the rest of the file

                } catch (EOFException e) {

                    done = true;

                }

            }

        } finally {

            writer.close();

        }

    }



    private static String cleanEntry(String entry) {

        String result = "";

        result = entry.replaceAll("\\nnull","");

        int index = result.indexOf("\n");

        if (index == 0) {

            result = result.substring(1);

        }

        return result + "\n";

    }



    private static byte skipHeader(DataInputStream f) throws IOException {

        f.readByte(); // 00: BE, BF: LE

        f.readByte();

        f.readInt();

        byte signature[] = new byte[10];

        f.read(signature);

        f.readUTF();

        byte padding[] = new byte[3];

        f.read(padding);

        byte amfVersion = f.readByte();

        return amfVersion;

    }



    private static int readU29AValue(DataInputStream f) throws IOException {

        int val = 0;

        byte b = f.readByte();

        if ((b & 0x80) != 0) {

            val = b & 0x7f;

            b = f.readByte();

        }

        if ((b & 0x80) != 0) {

            b &= 0x7f;

            val <<= 7;

            val |= b;

            b = f.readByte();

        }

        if ((b & 0x80) != 0) {

            b &= 0x7f;

            val <<= 7;

            val |= b;

            b = f.readByte();

        }

        val <<= 7;

        val |= b;

        return val;

    }



}