/***********************************************************
 * Copyright 2014 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.core.utils;

import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;

import javax.naming.NamingEnumeration;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Network utilities class.
 *
 * @author tproenca (Tiago Proenca)
 */
public class NetworkUtils {

    /** Caches the local ip address. */
    private static String localIpAddr = null;

    /** Caches the local ip address. */
    private static Map<String, String> nsLookupMap = new HashMap<String, String>();

    /** Logger utility. */
    private static final Logger logger = LoggerFactory.getLogger(NetworkUtils.class);

    /**
     * Returns the local ip address (not loopback)
     */
    public static synchronized String getLocalIpAddress() {
        if (localIpAddr == null) {
            try {
                Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
                while (interfaces.hasMoreElements()) {
                    NetworkInterface current = interfaces.nextElement();
                    if (!current.isUp() || current.isLoopback() || current.isVirtual())
                        continue;
                    Enumeration<InetAddress> addresses = current.getInetAddresses();
                    while (addresses.hasMoreElements()) {
                        InetAddress current_addr = addresses.nextElement();
                        if (!current_addr.isLoopbackAddress() && !(current_addr instanceof Inet6Address)) {
                            localIpAddr = current_addr.getHostAddress();
                            break;
                        }
                    }
                }
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
            }
        }
        return localIpAddr;
    }

    /**
     * Runs an nslookup to obtain the DNS name from a give IP address.
     *
     * @param ip
     *            The IP address.
     * @return The DNS name.
     */
    public static synchronized String nslookup(String ip) {
        String result = nsLookupMap.get(ip);

        if (result == null) {
            String[] bytes = ip.split("\\.");
            if (bytes.length == 4) {
                Hashtable<String, String> env = new Hashtable<String, String>();
                env.put("java.naming.factory.initial", "com.sun.jndi.dns.DnsContextFactory");

                try {
                    DirContext ctx = new InitialDirContext(env);

                    String reverseDnsDomain = bytes[3] + "." + bytes[2] + "." + bytes[1] + "." + bytes[0] + ".in-addr.arpa";
                    Attributes attrs = ctx.getAttributes(reverseDnsDomain, new String[] { "PTR" });

                    for (NamingEnumeration<? extends javax.naming.directory.Attribute> ae = attrs.getAll(); ae.hasMoreElements();) {
                        Attribute attr = ae.next();
                        String attrId = attr.getID();
                        for (Enumeration<?> vals = attr.getAll(); vals.hasMoreElements();) {
                            String value = vals.nextElement().toString();

                            if ("PTR".equals(attrId)) {
                                int len = value.length();
                                if (value.charAt(len - 1) == '.') {
                                    // Strip out trailing period
                                    value = value.substring(0, len - 1);
                                }
                                nsLookupMap.put(ip, value);
                                result = value;
                                break;
                            }
                        }
                    }
                    ctx.close();
                } catch (Exception e) {
                    logger.error(e.getMessage(), e);
                }
            }
        }
        return result;
    }
}
